# -*- Mode: Python; tab-width: 4 -*-

import windc
import winfont
import wingdi
import winwin
import layout
import mvc_scrollbar

import string

# This is only halfway converted to MVC: It uses MVC scrollbars,
# but does not use any kind of model for the data.  We may want
# to look into using mvc_seq_view instead.

SPLIT_WORD = winwin.SPLIT_WORD

class cell_scroll_window (winwin.python_window):

	style=winwin.WS_CHILD | winwin.WS_VISIBLE

	last_scroll_position = 0, 0
	scroll_models = None, None

	def create (*args):
		self = apply (winwin.python_window.create, args)

		vm, hm = self.scroll_models
		if vm is None:
			vm = mvc_scrollbar.scroll_model()
			vm.add_view (self)
		if hm is None:
			hm = mvc_scrollbar.scroll_model()
			hm.add_view (self)
		
		self.scroll_models = vm, hm

		return self

	def begin_paint (self, paintstruct=None):
		dc = winwin.python_window.begin_paint (self, paintstruct)
		self.prepare_dc (dc)
		return dc

	def prepare_dc (self, dc):
		vm, hm = self.scroll_models
		cx, cy = self.get_cell_size()
		dc.set_window_org (cx * hm.position, cy * vm.position)

	def get_dc (self):
		vm, hm = self.scroll_models
		cx, cy = self.get_cell_size()
		dc = winwin.python_window.get_dc (self)
		dc.set_window_org (cx * hm.position, cy * vm.position)
		return dc

	def dp_to_lp (self, (x,y)):
		dx, dy = self.get_position()
		return x + dx, y + dy

	def lp_to_dp (self, (x,y)):
		dx, dy = self.get_position()
		return (x - dx, y - dy)

	def WM_SIZE (self, wparam, lparam):
		self.size = SPLIT_WORD (lparam)
		w, h = self.size
		cx, cy = self.get_cell_size()
		vm, hm = self.scroll_models
		hm.page = w / cx
		vm.page = h / cy

	def notify (self, model, hint):
		vm, hm = self.scroll_models
		lx, ly = self.last_scroll_position
		cx, cy = self.get_cell_size()
		nx, ny = hm.position, vm.position
		nx = nx * cx
		ny = ny * cy
		
		if model is vm:
			dx, dy = 0, ly - ny
		elif model is hm:
			dx, dy = lx - nx, 0

		self.scroll_window (dx, dy)
		self.last_scroll_position = nx, ny
		self.update_window()

class text_scroll_window (cell_scroll_window):
	
	font_height = 16
	font = winfont.font ('Courier New', height=font_height).create()
	cell_size = None
	lines = []

	def get_cell_size (self):
		if self.cell_size is None:
			# don't call ours, it needs cell_size!
			dc = winwin.python_window.get_dc (self)
			dc.select_object (self.font)
			tm = dc.get_text_metrics()
			dc.release_dc()
			self.cell_size = tm.ave_char_width, tm.height

		return self.cell_size

	def WM_PAINT (self, wparam, lparam):
		dc = self.begin_paint()
		self.do_paint (dc)
		self.end_paint()

	def do_paint (self, dc):

		dc.select_object (self.font)
		dc.set_bk_color()
		dc.set_bk_mode (windc.TRANSPARENT)
		#dc.set_text_align (wingdi.TA_UPDATECP)

		l,t,r,b = dc.get_clip_box()

		# determine the range of lines we need to draw

		fw, fh = self.get_cell_size()
		first = t / fh
		last  = b / fh
		if b % fh:
			last = last + 1

		last = min (len(self.lines), last)

		for i in range (first, last):
			self.draw_line (dc, i*fh, i)

	def draw_line (self, dc, y, index):
		text = self.lines[index]
		dc.text_out ((0,y), text)


class text_edit_window (text_scroll_window):
	
	current_pos = (0,0)

	def WM_CHAR (self, wparam, lparam):
		# insert / overwrite, etc...

		fw, fh = self.get_cell_size()

		ch = chr (wparam)
		w, h = self.size
		x, y = self.current_pos

		line = self.lines[y]

		if ch == '\r':
			if x < len(line):
				self.lines[y] = line[:x]
				self.lines.insert (y+1, line[x:])
			else:
				self.lines.insert (y+1, '')
			self.current_pos = 0, y+1
			self.invalidate_rect ((0,y * fh,w,h))
		else:
			chars = map (None, line)
			chars.insert (x, ch)
			self.lines[y] = string.join (chars, '')
			x = x + 1
			cw = self.font_width
			self.current_pos = x, y
			self.invalidate_rect (
				((x-1) * cw , y * fh, w, (y+1) * fh)
				)

		self.set_caret_pos()

		return 1

	def invalidate_line_range (self, low, high):
		w, h = self.size
		fh = self.font_height
		self.invalidate_rect ((0, low*fh, w, high*fh))

	def WM_DESTROY (self, wparam, lparam):
		winwin.user32.PostQuitMessage (0)

	def WM_SETFOCUS (self, wparam, lparam):
		self.show_caret()
	
	def show_caret (self):
		cw,ch = self.get_cell_size()
		user32.CreateCaret (self, 0, cw, ch)
		self.set_caret_pos()
		user32.ShowCaret (self)

	def set_caret_pos (self):
		cw,ch = self.get_cell_size()
		x, y = self.current_pos
		x, y = self.lp_to_dp ((x * cw, y * ch))
		user32.SetCaretPos (x, y)

	def WM_KILLFOCUS (self, wparam, lparam):
		user32.DestroyCaret (self)

	def WM_LBUTTONDOWN (self, wparam, lparam):
		mx, my = self.dp_to_lp (SPLIT_WORD (lparam))
		cx, ch = self.get_cell_size()
		x = mx / cx
		y = my / ch
		y = min (len (self.lines) - 1, y )
		x = min (len (self.lines[y]), x)
		self.current_pos = x, y
		self.set_caret_pos ()

class text_component (layout.container):
	
	style = winwin.WS_CHILD | winwin.WS_VISIBLE

	child_window_class = text_scroll_window

	def create (*args):
		self = apply (layout.container.create, args)
		hm = mvc_scrollbar.scroll_model()
		vm = mvc_scrollbar.scroll_model()
		tw = self.child_window_class (parent=self, scroll_models = (vm, hm)).create()
		bl = layout.border_layout (self)
		vs = mvc_scrollbar.vertical_scrollbar_component (model=vm, parent=self).create()
		hs = mvc_scrollbar.horizontal_scrollbar_component (model=hm, parent=self).create()

		bl.add (vs, 'east')
		bl.add (hs, 'south')
		bl.add (tw, 'center')

		hm.add_view (tw)
		vm.add_view (tw)

		self.text_window = tw
		return self

	def set_contents (self, lines):

		def clean (line):
			# remove CR, LF
			while line and line[-1] in '\r\n':
				line = line[:-1]
			# replace tabs
			return string.join (string.split (line, '\t'), '    ')

		tw = self.text_window
		tw.lines = map (clean, lines)
		vm, hm = tw.scroll_models

		if tw.lines:
			vm.limit = len (tw.lines)
			hm.limit = max (map (len, tw.lines))
		else:
			vm.limit = 0
			hm.limit = 0

class text_edit_component (text_component):
	
	child_window_class = text_edit_window

def test():
	lines = open ('cell_scroll.py', 'r').readlines()
	w = text_edit_component ('text scroll test', style=winwin.WS_OVERLAPPEDWINDOW).create()
	w.set_contents (lines)
	w.show_window()

if __name__ == '__main__':
	import msgloop
	test()
	msgloop.go()
