# -*- Mode: Python; tab-width: 4 -*-

import layout
import quadtree
import winwin
import windll
import dynscroll
import mvc_scrollbar
import windc
import wingdi

gdi32 = windll.module ('gdi32')
SPLIT_WORD = winwin.SPLIT_WORD

import winfont


# ===========================================================================
# Tree Traversal
# ===========================================================================

import newdir
import types

interesting_types = (
	types.ClassType,
	types.ModuleType,
	types.FunctionType,
	types.MethodType,
	)

def get_children (x):
	"Get child attributes of <x>, paying attention to only certain types"

	if type(x) in (types.ClassType, types.ModuleType):
		r = []
		for name in newdir.dir (x):
			ob = getattr (x, name)
			if type(ob) in interesting_types:
				r.append (ob)
		return r
	else:
		return []

# ===========================================================================
# Graphical Objects
# ===========================================================================

class labeled_object:

	font_height = 14
	font = winfont.font ('Arial', height=font_height).create()

	type_brushes = {
		types.ModuleType:	(255, 128, 128),
		types.ClassType:	(128, 255, 128),
		types.MethodType:	(128, 128, 255),
		types.FunctionType:	(128, 255, 255),
		}

	for k,v in type_brushes.items():
		r,g,b = v
		type_brushes[k] = wingdi.brush (red=r,green=g,blue=b).create()

	def __init__ (self, dc, pos, object):
		x, y = pos
		self.text = text = object.__name__
		# compute our bounding rectangle
		dc.select_object (self.font)
		w, h = dc.get_text_extent (text)
		self.rect = (
			x, y-((h+8)/2),
			x+w+8, y+((h+8)/2)
			)
		self.text_pos = x+4, y-(h/2)
		self.brush = self.type_brushes[type(object)]
		return 

	def move (self, dx, dy):
		# don't call this unless you first remove it from the quadtree.
		tx, ty = self.text_pos
		self.text_pos = tx+dx, ty+dy
		l,t,r,b = self.rect
		self.rect = l+dx, t+dy, r+dx, b+dy

	def get_rect (self):
		return self.rect

	def do_paint (self, dc):
		dc.set_bk_color()
		dc.set_bk_mode ()
		r = self.rect
		dc.select_object (self.brush)
		dc.rectangle (r)
		dc.select_object (self.font)
		dc.text_out (self.text_pos, self.text)

def normalize (x1,y1,x2,y2):
	if x1 > x2:
		x1, x2 = x2, x1
	if y1 > y2:
		y1, y2 = y2, y1
	return x1, y1, x2, y2

class line:

	pen = wingdi.pen().create()

	def __init__ (self, p1, p2):
		self.points = p1, p2
		x1, y1 = p1
		x2, y2 = p2
		self.rect = normalize (x1, y1, x2, y2)

	def get_rect (self):
		return self.rect
		
	def do_paint (self, dc):
		p1, p2 = self.points
		dc.select_object (self.pen)
		dc.move_to (p1)
		dc.line_to (p2)

# ===========================================================================

class cell:
	def __init__ (self, value):
		self.value = value

class tree_window (dynscroll.scroll_window):

	offset = 80
	captured = None

	def create (*args):
		self = apply (dynscroll.scroll_window.create, args)
		self.quadtree = qt = quadtree.quadtree ((0,0,256,256))
		return self

	# ===========================================================================
	# Painting
	# ===========================================================================

	def WM_SIZE (self, wparam, lparam):
		w, h = self.size = SPLIT_WORD (lparam)
		vm, hm = self.scroll_models
		# match the scroll ranges to the quadtree's bounds
		l,t,r,b = self.quadtree.get_bounds()
		self.origin = l,t
		hm.limit = (r-l)+20
		hm.page = w
		vm.limit = (b-t)+20
		vm.page = h

	def prepare_dc (self, dc):
		# adjust our origin to the quadtree's bounds
		vm, hm = self.scroll_models
		ox, oy = self.origin
		dc.set_window_org (hm.position + ox, vm.position + oy)

	def WM_PAINT (self, wparam, lparam):
		dc = self.begin_paint()
		clip = dc.get_clip_box()
		self.quadtree.search_apply (clip, lambda x,dc=dc: x.do_paint (dc))
		self.end_paint()

	# ===========================================================================
	# Mouse Input
	# ===========================================================================

	def WM_LBUTTONDOWN (self, wparam, lparam):
		self.set_capture()
		x, y = SPLIT_WORD (lparam)
		vm, hm = self.scroll_models
		ox, oy = x + hm.position, y + vm.position
		self.captured = ox, oy

	def WM_MOUSEMOVE (self, wparam, lparam):
		if self.captured is not None:
			x, y = SPLIT_WORD (lparam)
			ox, oy = self.captured
			vm, hm = self.scroll_models
			hm.position = ox - x
			vm.position = oy - y

	def WM_LBUTTONUP (self, wparam, lparam):
		self.release_capture()
		self.captured = None
			
	# ===========================================================================
	# Object Layout
	# ===========================================================================

	def get_output_point (self, rect):
		l,t,r,b = rect
		return r, t+((b-t)/2)

	def layout (self, dc, root, x, y_cell, seen):
		qt = self.quadtree
		children = []
		for child in get_children (root):
			# make sure we don't repeat ourselves...
			if not seen.has_key(id (child)):
				children.append (child)
				seen[id(child)] = 1

		gob = labeled_object (dc, (0,0), root)
		op = self.get_output_point (gob.get_rect())

		if children:
			oy = y_cell.value
			output_points = []
			cx = x + op[0] + self.offset
			for child in children:
				output_points.append (
					self.layout (dc, child, cx, y_cell, seen)
					)

			min_y = min (output_points)
			max_y = max (output_points)
			h = max_y - min_y
			ry = min_y + (h/2)
			gob.move (x, ry)
			qt.insert (gob)

			for oy in output_points:
				qt.insert (line ((cx, oy),(x+op[0],ry)))

			return ry
		else:
			y = y_cell.value
			gob.move (x, y)
			qt.insert (gob)
			y_cell.value = y + ((5 * labeled_object.font_height) / 2)
			return y

class tree_component (layout.border_container):

	style = winwin.WS_CHILD | winwin.WS_VISIBLE

	vert_scrollbar_border = 'east'
	horz_scrollbar_border = 'south'

	def create (*args):
		self = apply (layout.border_container.create, args)
		tw = tree_window (parent=self).create()
		vm, hm = tw.scroll_models		
		vs = mvc_scrollbar.vertical_scrollbar_component (model=vm, parent=self,  width=14).create()
		hs = mvc_scrollbar.horizontal_scrollbar_component (model=hm, parent=self, width=14).create()
		self.add (vs, self.vert_scrollbar_border)
		self.add (hs, self.horz_scrollbar_border)
		self.add (tw, 'center')
		vm.add_view (tw)
		hm.add_view (tw)
		self.tree_window = tw
		return self
		
	def get_preferred_size (self):
		# stretchable in both directions
		return sys.maxint, sys.maxint

def test ():
	w = tree_component ('Module Tree Demo - Drag Window With Mouse', style=winwin.WS_OVERLAPPEDWINDOW).create()
	import sys
	tw = w.tree_window
	print 'Laying out object tree...'
	dc = tw.get_dc()
	tw.layout (dc, sys.modules['__main__'], 20, cell(500), {})
	#tw.layout (dc, winwin, 20, cell(500), {})
	dc.release_dc()
	w.show_window()

if __name__ == '__main__':
	import msgloop
	test()
	msgloop.go()
