# -*- Mode: Python; tab-width: 4 -*-

# Overlapped I/O, ReadFileEx, WriteFileEx, etc...

# This demonstrates using completion routines.  Another technique
# uses event objects.  Two different ways of doing the same thing.
# MS calls completion routines 'Asynchronous Procedure Calls'.  This
# is a bit misleading.  An APC is nothing more than a function pointer
# in a queue.  It is not actually called asynchronously, but only when
# you 'enter an alertable wait state', with a function like
# WaitForMultipleObjects.

# The same affect can be achieved 'manually' by using Event objects.
# 1) Create An Event Object
# 2) Stuff it into an OVERLAPPED structure
# 3) Call ReadFileEx, WriteFileEx, AcceptEx, etc...
# 4) Your main WaitForMultipleObjects loop should poll all your
#    outstanding Events.
# 5) Scan through your events looking for those that have been triggered.

# Note that AcceptEx does not support completion routines.

# ===========================================================================
#			   Overlapped I/O with completion routines.
# ===========================================================================

# Note: It is VERY IMPORTANT to have the completion routine:
#
# 1) return an integer
# 2) not directly raise an error
#
# Otherwise python will exit on a fatal no-current-thread condition.
# Once I understand this a bit better I hope to remove the latter
# restriction, and make the whole thing a bit more safe.

import windll
import winerror
import gencb
import structob
import npstruct

import thread

kernel32 = windll.module ('kernel32')

class OVERLAPPED (structob.struct_object):
	oracle = structob.Oracle (
		'Overlapped I/O object',
		'Nlllll',
		('Internal',
		 'InternalHigh',
		 'Offset',
		 'OffsetHigh',
		 'hEvent'
		 )
		)

class async_file:

	def __init__ (self, handle, buffer_size = 1024):

		self.buffer = windll.membuf (1024)
		self.handle = handle
		self.overlapped = OVERLAPPED()
		self.read_completion_callback = gencb.generated_callback ('lll', self.completion_read)
		self.write_completion_callback = gencb.generated_callback ('lll', self.completion_write)
		print self.handle

	def completion_read (self, error_code, num_bytes, overlapped):
		print error_code, num_bytes, overlapped
		data = self.buffer.read (0, num_bytes)
		print 'Read: %d %s' % (thread.get_ident(), repr(data))
		#self.write (data)
		self.overlapped.describe()
		# re-trigger the read event
		self.read ()
		return 0

	def completion_write (self, error_code, num_bytes, overlapped):
		print error_code, num_bytes, overlapped
		data = self.buffer.read (0, num_bytes)
		print 'Write: %d %s' % (thread.get_ident(), repr(data))
		return 0

	def read (self):
		result = kernel32.ReadFileEx (
			self.handle,
			self.buffer,
			len(self.buffer),
			self.overlapped,
			self.read_completion_callback
			)
		if not result:
			print kernel32.GetLastError()
			print winerror.get_error_string()

	def write (self, data):
		self.buffer.write (data)
		result = kernel32.WriteFileEx (
			self.handle,
			self.buffer,
			len(data),
			self.overlapped,
			self.read_completion_callback
			)
		if not result:
			print kernel32.GetLastError()
			print winerror.get_error_string()

# from <winbase.h>
STD_INPUT_HANDLE    = -10
STD_OUTPUT_HANDLE   = -11
STD_ERROR_HANDLE    = -12


def loop():
	print 'entering SleepEx loop...', thread.get_ident()
	i = 0
	while 1:
		# sleep 5 seconds in an alertable wait state.
		print 'Sleep: %d' % i
		kernel32.SleepEx(5000, 1)
		i = i + 1

# This doesn't appear to work, my guess is that the STD files
# are not flagged for overlapped i/o.
def test():
	stdin = async_file (kernel32.GetStdHandle (STD_INPUT_HANDLE))
	stdout = async_file (kernel32.GetStdHandle (STD_OUTPUT_HANDLE))
	stdin.stdout = stdout
	stdin.read (4)
	loop()

# I _think_ this approach can be integrated into the 'asyncore'
# paradigm, maybe with a wrapper class? (so that a call to recv()
# will trigger a call to ReadFileEx, etc...)  Then the only requirement
# for the application is to use an alertable wait function, like
# MsgWaitForMultipleObjects, instead of GetMessage.
# However, this still does not address the issue of listening sockets.
# [this is probably why AcceptEx came into existence]

import socket

def server():
	# create a server socket
	s = socket.socket (socket.AF_INET, socket.SOCK_STREAM)
	s.bind (('127.0.0.1', 3434))
	s.listen (5)
	return s.accept()

def async_server_test():
	s = socket.socket (socket.AF_INET, socket.SOCK_STREAM)
	s.bind (('127.0.0.1', 3434))
	s.listen (5)
	f = async_file (s.fileno())
	f.read()
	loop()

def client():
	s = socket.socket (socket.AF_INET, socket.SOCK_STREAM)
	s.connect (('127.0.0.1', 3434))
	f = async_file (s.fileno())
	f.read()
	loop()

#if __name__ == '__main__':
#	client()

# ===========================================================================
#			   Overlapped I/O with Event Objects
# ===========================================================================

# Need an object wrapper for ReadFileEx, WriteFileEx, AcceptEx, etc...

# Is there no way to actually read the status of an Event Object?

class event:

	def __init__ (self, manual_reset=0, initial_state=0, name=0):
		self.handle = kernel32.CreateEvent (0, manual_reset, initial_state, name)

	def __int__ (self):
		return self.handle

# OVERLAPPED->Internal indicates status.

overlapped_map = {}

class overlapped:

	def __init__ (self, handle):
		self.handle = handle
		self.buffers = windll.membuf (1024), windll.membuf (1024)
		ie, oe = self.events = event(), event()
		io, oo = self.overlaps = OVERLAPPED(), OVERLAPPED()
		io.hEvent = ie.handle
		oo.hEvent = oe.handle
		overlapped_map[handle] = self
		
	def __int__ (self):
		return self.handle

	def read (self, num_bytes):
		ib, ob = self.buffers
		io, oo = self.overlaps
		return kernel32.ReadFile (self, ib, num_bytes, 0, io)
			
	def write (self, data):
		ib, ob = self.buffers
		io, oo = self.overlaps
		return kernel32.ReadFile (self, ob, num_bytes, 0, oo)


# Optimization: We could place active objects at the tail end of the
# array, since WFMO returns the index of the lowest-numbered
# triggering event.

WAIT_TIMEOUT = 0x102
STILL_ACTIVE = 0x103

def poll (timeout=30000):
	handles = []
	objects = overlapped_map.values()
	for x in objects:
		ie, oe = x.events
		handles.append (ie.handle)
		handles.append (oe.handle)

	n = len(handles)
	array = windll.membuf (npstruct.pack ('l'*n, tuple(handles)))
	index = kernel32.WaitForMultipleObjects (n, array, 0, timeout)

	if index != WAIT_TIMEOUT:
		# scan through the objects, starting at index.
		# if (OVERLAPPED->Internal != STATUS_PENDING) then the
		# operation has completed.  [see the win32
		# HasOverlappedIoCompleted macro]
		for x in objects[index:]:
			io, oo = x.overlaps
			print 'channel: %d' % x.handle,
			print 'input: ', io.Internal,
			print 'output: ', oo.Internal
			
def client():
	s = socket.socket (socket.AF_INET, socket.SOCK_STREAM)
	s.connect (('127.0.0.1', 3434))
	f = overlapped (s.fileno())
	f.read (1024)
	while 1:
		poll()
