# -*- Mode: Python; tab-width: 4 -*-

# An experiment: is it 'overkill' to use MVC with a layout manager?
# For example, a sequence model can be used with flow layout, so that
# components can be inserted, removed, etc... from a container window,
#
# The layout code is relatively simple, but actually _using_ this
# seems a bit complex (see the test function below).  I'm not sure yet
# whether this complexity is worth it.  Perhaps a nicer programming
# interface for all of these will change my mind.
#

import wingdi
import winwin
import mvc

# returns LOWORD, HIWORD
def SPLIT_WORD (x):
	return (x&0xffff, (x>>16))

class container_mixin:

	def add (self, child, constraint=None):
		self.layout_manager.add (child, constraint)

	def WM_SIZE (self, wparam, lparam):
		if lparam:
			self.layout_manager.do_layout()

class border_model (mvc.auto_model):

	def __init__ (self, north=None, south=None, east=None, west=None, center=None):
		mvc.auto_model.__init__ (
			self,
			north	= north,
			south	= south,
			east	= east,
			west	= west,
			center	= center,
			)

class border_layout (mvc.view):

	def __init__ (self, container):
		self.container = container
		self.container.layout_manager = self # cycle

	def notify (self, model, hint):
		self.do_layout()

	def do_layout (self):
		# Lay out each component.  Determine the bounds of the center
		# rectangle, given the preferred sizes of any child windows.
		#
		#   +--------------+
		#   |              |
		#  	+---o------o---+          cl, ct       cr, ct
		#  	|	|  	   |   |
   	   	#	|	|  	   |   |
		#	|	|  	   |   |
		#	+---o------o---+          cl, cb       cr, cb
		#	|			   |
		# 	+--------------+
		#

		def get_size (child):
			if child:
				return child.get_preferred_size()
			else:
				return (0,0)

		def move_child (child, l,t,r,b):
			if child:
				child.move_window (l,t,r,b)

		l,t,r,b = self.container.get_client_rect()

		north	= self.model.north
		south	= self.model.south
		east	= self.model.east
		west	= self.model.west
		center	= self.model.center

		ct = t + get_size (north) [1]
		cb = b - get_size (south) [1]
		cl = l + get_size (west) [0]
		cr = r - get_size (east) [0]

		move_child (north, l, t, r-l, ct-t)
		move_child (south, 0, cb, r-l, b-cb)
		move_child (west, 0, ct, cl-l, cb-ct)
		move_child (east, cr, ct, r-cr, cb-ct)
		move_child (center, cl, ct, cr-cl, cb-ct)

class flow_layout (mvc.view):

	# use a sequence model with this layout manager.

	def __init__ (self, container):
		self.container = container
		self.container.layout_manager = self # cycle

	def notify (self, model, hint):
		self.do_layout()

	def do_layout (self):
		rect = (l,t,r,b) = self.container.get_client_rect()
		x, y = l,t

		i = 0

		while i < len(self.model):

			row = []
			tallest = 0

			while i < len(self.model):
				child = self.model[i]
				w, h = child.get_preferred_size()
				# pack until we run out of room on the right
				if ((w + x) > r) and (x != l) :
					break
				else:
					row.append ((child, (w, h)))

				if h > tallest:
					tallest = h

				i = i + 1
				x = x + w

			x = l
			# lay out this row
			for child, (w, h) in row:
				child.move_window (x, y + ((tallest - h) / 2), w, h)
				x = x + w

			y = y + tallest
			x = l

class container (winwin.python_window, container_mixin):

	# If your layout manager will completely cover the container
	# window, then set this to one.  Otherwise, set it to zero.
	handle_erase_background = 1

	def WM_ERASEBKGND (self, wparam, lparam):
		return self.handle_erase_background

frame = container

def test():
	import mvc_button
	# create a couple of button models.
	bm1 = mvc_button.push_button_model (label='one')
	bm2 = mvc_button.push_button_model (label='two')
	# create a container
	c = frame ('MVC test')
	c.handle_erase_background = 0
	# create a sequence model
	sm = mvc.sequence_model()
	# create a layout manager
	lm = flow_layout (c)
	# attach the model to the manager
	lm.set_model (sm)
	# create the button windows
	bw1 = mvc_button.text_button ('', parent=c)
	bw2 = mvc_button.text_button ('', parent=c)
	# attach the button models
	bw1.set_model (bm1)
	bw2.set_model (bm2)
	# add them to the container
	sm.append (bw1)
	sm.append (bw2)
	# create the windows
	c.create()
	bw1.create().show_window()
	bw2.create().show_window()
	c.show_window()
	return c

if __name__ == '__main__':
	import msgloop
	test()
	msgloop.go()
