# -*- Mode: Python; tab-width: 4 -*-

#
# Simple layout manager support, roughly modeled after the java awt.
#
# Each container window implements a layout manager.  The layout
# manager will position the children according to its policy.
#
# Currently implemented:
#   border_layout (north, south, east, west, center)
#   flow_layout (left-to-right, top-to-bottom)
#   hbox and vbox implement Fresco/TeX-style layout, with stretchable
#     glue and alignment.

# TODO:
#   proportional_layout (like that managed by splitter.py)

# This is still slightly mis-implemented; AWT has the layout manager
# as a separate object.  However, it's probably not worth doing, since
# the only realistic use for this would be to write slightly different
# versions of particular constraint styles.

# TODO: lightweight containers: containers that 'borrow' their win32
# window from their parent.  I did this with the previous brain-dead
# layout classes, and it worked; keep thinking about it.

import wingdi
import winwin

# returns LOWORD, HIWORD
def SPLIT_WORD (x):
	return (x&0xffff, (x>>16))

class border_container (winwin.python_window):

	def create (*args):
		self = apply (winwin.python_window.create, args)
		self.layout_children = {}
		return self

	def WM_SIZE (self, wparam, lparam):
		self.size = SPLIT_WORD (lparam)
		self.do_layout()

	constraints = ('north', 'south', 'east', 'west', 'center')

	def add (self, child, constraint):
		if constraint not in self.constraints:
			raise ValueError, "Unsupported constraint.  Use one of %s" % repr(self.constraints)
		elif self.layout_children.has_key (constraint):
			raise ValueError, "A child has already been positioned there"
		else:
			self.layout_children[constraint] = child

	def get_child (self, border):
		if self.layout_children.has_key (border):
			return self.layout_children[border]
		else:
			return None

	def get_child_size (self, border):
		child = self.get_child (border)
		if child :
			return child.get_preferred_size()
		else:
			return (0, 0)

	def move_child (self, child, (l,t,w,h)):
		child.move_window (l,t,w,h)

	def do_layout (self):
		# Lay out each component.  Determine the bounds of the center
		# rectangle, given the preferred sizes of any child windows.
		#
		#   +--------------+
		#   |              |
		#  	+---o------o---+          cl, ct       cr, ct
		#  	|	|  	   |   |
   	   	#	|	|  	   |   |
		#	|	|  	   |   |
		#	+---o------o---+          cl, cb       cr, cb
		#	|			   |
		# 	+--------------+
		#

		l,t,r,b = self.get_client_rect()

		ct = t + self.get_child_size ('north')[1]
		cb = b - self.get_child_size ('south')[1]
		cl = l + self.get_child_size ('west')[0]
		cr = r - self.get_child_size ('east')[0]

		child = self.get_child ('north')
		if child:
			self.move_child (child, (l,t,r-l,ct-t))

		child = self.get_child ('south')
		if child:
			self.move_child (child, (0,cb,r-l,b-cb))
			
		child = self.get_child ('west')
		if child:
			self.move_child (child, (0,ct,cl-l,cb-ct))
		
		child = self.get_child ('east')
		if child:
			self.move_child (child, (cr,ct,r-cr,cb-ct))

		child = self.get_child ('center')
		if child:
			self.move_child (child, (cl,ct,cr-cl,cb-ct))


class flow_container (winwin.python_window):

	def create (*args):
		self = apply (winwin.python_window.create, args)
		self.layout_children = []
		return self

	def WM_SIZE (self, wparam, lparam):
		self.size = SPLIT_WORD (lparam)
		self.do_layout()

	def add (self, child, constraint=None):
		# no constraints
		self.layout_children.append (child)

	def move_child (self, child, (l,t,w,h)):
		child.move_window (l,t,w,h)

	def do_layout (self):

		rect = (l,t,r,b) = self.get_client_rect()
		x, y = l,t

		i = 0

		while i < len(self.layout_children):

			row = []
			tallest = 0

			while i < len(self.layout_children):
				child = self.layout_children[i]
				w, h = child.get_preferred_size()
				# pack until we run out of room on the right
				if ((w + x) > r) and (x != l) :
					break
				else:
					row.append ((child, (w, h)))

				if h > tallest:
					tallest = h

				i = i + 1
				x = x + w

			x = l
			# lay out this row
			for child, (w, h) in row:
				self.move_child (child, (x, y + ((tallest - h) / 2), w, h))
				x = x + w

			y = y + tallest
			x = l

import sys

class hbox (winwin.python_window):

	style = winwin.WS_CHILD

	# glue should probably just be an instance of <spacer>
	GLUE = 'glue'

	class spacer:
		def __init__ (self, width, height):
			self.size = width, height

		def get_preferred_size (self):
			return self.size

		def move_window (*args):
			pass

	GLUE = spacer (sys.maxint, 1)

	def create (*args):
		self = apply (winwin.python_window.create, args)
		self.layout_children = []
		return self

	def WM_SIZE (self, wparam, lparam):
		self.size = SPLIT_WORD (lparam)
		self.do_layout()

	def add (self, child, constraint=0):
		# <constraint> is the alignment value:
		#   {-1:left|top, 0:centered, 1:right|bottom} 
		self.layout_children.append ((child, constraint))

	def move_child (self, child, (l,t,w,h)):
		child.move_window (l,t,w,h)

	def do_layout (self):

		l,t,r,b = self.get_client_rect()
		
		w, h = r-l, b-t
		tw = 0
		glues = 0
		# find the non-glue, non-stretchable children
		for c, a in self.layout_children:
			cw, ch = c.get_preferred_size()
			if cw is not sys.maxint:
				tw = tw + cw
			else:
				glues = glues + 1
		
		# leftover
		left = w - tw
		if left < 0:
			left = 0

		# divide the remaining room up evenly
		if glues:
			spacers = left / glues
		else:
			spacers = 0

		x = 0
		for c, a in self.layout_children:
			if c is not self.GLUE:
				cw, ch = c.get_preferred_size()
				if cw is sys.maxint:
					cw = spacers
				if ch is sys.maxint:
					ch = h
				if a < 0:
					pos = (x, 0, cw, ch)
				elif a > 0:
					pos = (x, h-ch, cw, ch)
				else:
					pos = (x, (h-ch)/2, cw, ch)

				self.move_child (c, pos)
				x = x + cw
			else:
				x = x + spacers


	def get_preferred_size (self):
		tw = 0
		mh = 0
		for c, a in self.layout_children:
			if c is not self.GLUE:
				cw, ch = c.get_preferred_size()
				mh = max (ch, mh)
				if cw is not sys.maxint:
					tw = tw + cw
				else:
					tw = sys.maxint
					break
		return tw, mh

class vbox (hbox):

	def do_layout (self):

		l,t,r,b = self.get_client_rect()
		
		w, h = r-l, b-t
		th = 0
		glues = 0
		# find the non-glue, non-stretchable children
		for c, a in self.layout_children:
			if c is not self.GLUE:
				cw, ch = c.get_preferred_size()
				if ch is not sys.maxint:
					th = th + ch
				else:
					glues = glues + 1
			else:
				glues = glues + 1
		
		# leftover
		left = h - th
		if left < 0:
			left = 0

		# divide the remaining room up evenly
		if glues:
			spacers = left / glues
		else:
			spacers = 0

		y = 0
		for c, a in self.layout_children:
			if c is not self.GLUE:
				cw, ch = c.get_preferred_size()
				if ch is sys.maxint:
					ch = spacers
				if cw is sys.maxint:
					cw = w
				if a < 0:
					pos = (0, y, cw, ch)
				elif a > 0:
					pos = (w-cw, y, cw, ch)
				else:
					pos = ((w-cw)/2, y, cw, ch)

				self.move_child (c, pos)
				y = y + ch
			else:
				y = y + spacers

	def get_preferred_size (self):
		th = 0
		mw = 0
		for c, a in self.layout_children:
			if c is not self.GLUE:
				cw, ch = c.get_preferred_size()
				mw = max (cw, mw)
				if ch is not sys.maxint:
					th = th + ch
				else:
					th = sys.maxint
					break
		return mw, th

def test_hbox():
	h = hbox(style=winwin.WS_OVERLAPPEDWINDOW).create()
	import mvc_button
	b1 = mvc_button.make_text_button ('Hello')
	b2 = mvc_button.make_text_button ('There')
	b3 = mvc_button.make_text_button ('High')
	b4 = mvc_button.make_text_button ('Low')

	for b in (b1,b2,b3,b4):
		b.parent = h
		b.create()

	h.show_window()
	
	h.add (h.GLUE)

	for b,a in ((b1,0), (b2,0), (b3,-1), (b4,1)):
		h.add (b, a)
		h.add (h.GLUE)
		b.show_window()

	return h

if __name__ == '__main__':
	import msgloop
	test_hbox()
	msgloop.go()
