# -*- Mode: Python; tab-width: 4 -*-

# Implements support classes for the Model-View-Controller pattern.
#
# Inside:
#
# class model
#   base class for models
#
# class auto_model
#   automatic attribute-model class
#
# class dictionary_model
#   model for a dictionary interface
#
# class sequence_model
#   model for a sequence interface
#
# class view
#   base class for views
#
# class printing_view
#   helper class.  prints change notifications and their hints.

import string

import sys

class model:
	"base class for model component of model-view-controller pattern"
	
	def __init__ (self):
		self.views = []

	def change (self, hint=None):
		for view in self.views:
			view.notify (self, hint)

	def add_view (self, view):
		if view not in self.views:
			self.views.append (view)

	def remove_view (self, view):
		if view in self.views:
			self.views.remove (view)

# This is an 'auto' model object.  It conveniently captures the
# notify/change/get/set behavior needed by the model-view-controller
# pattern.  All _you_ need to do to create a model is to construct one
# using keyword arguments.

class auto_model (model):

	def __init__ (self, **init_properties):
		self.__dict__['properties'] = {}
		self.__dict__['views'] = []
		for k,v in init_properties.items():
			self.properties[k] = v

	def __getattr__ (self, attr):
		return self.properties[attr]

	def __setattr__ (self, property, value):
		old_value = self.properties[property]
		# check for a set hook
		method = 'set_%s_hook' % property
		if hasattr (self, method):
			value = apply (getattr (self, method), (value,))
			if value is None:
				print 'Warning: None returned by %s.%s' % (
					self.__class__.__name__,
					method
					)
		if old_value != value:
			self.properties[property] = value
			self.change (property)

	def __repr__ (self):
		r = []
		for k,v in self.properties.items():
			r.append ('%s=%s' % (k, repr(v)))
		r = string.join (r, ' ')
		return '<%s model (%s) at %x>' % (self.__class__.__name__, r, id(self))

	# needed for "<model> in <model-list>"
	def __cmp__ (self, other):
		return self is other

	def new_property (self, property, value=None):
		"dynamically add a new property to a model"
		self.properties[property] = value

class dictionary_model (model):

	# __getitem__ __setitem__ __delitem__
	def __init__ (self):
		model.__init__ (self)
		self.d = {}

	HINT_CHANGE = 'change'
	HINT_INSERT = 'insert'
	HINT_REMOVE = 'remove'
	
	# getters

	def __getitem__ (self, i):
		return self.d[i]
	
	def __len__ (self):
		return len (self.d)

	# setters
	
	def __setitem__ (self, i, v):
		if self.d.has_key (i):
			old_v = self.d[i]
			if v is not old_v:
				self.d[i] = v
				self.change ((self.HINT_CHANGE, i))
		else:
			self.d[i] = v
			self.change ((self.HINT_INSERT, i))

	def __delitem__ (self, i):
		del self.d[i]
		self.change ((self.HINT_REMOVE, i))

class sequence_model (model):

	def __init__ (self, seq=None):
		model.__init__ (self)
		if seq is None:
			self.seq = []
		else:
			self.seq = seq

	# notification types
	HINT_CHANGE = 'change' # low, high
	HINT_INSERT = 'insert' # start, number of items
	HINT_REMOVE = 'remove' # start, number of items

	# getters
	def __getitem__ (self, i):
		return self.seq[i]

	def __getslice__ (self, i, j):
		return self.seq[i:j]

	def __len__ (self):
		return len (self.seq)

	def __repr__ (self):
		return '<sequence model at %x %s>' % (id(self),self.seq)

	# setters

	def __setitem__ (self, i, v):
		if i < 0:
			i = len(self.seq) + i

		old_v = self.seq[i]
		if v is not old_v:
			self.seq[i] = v
			self.change ((self.HINT_CHANGE, i, i))

	def __setslice__ (self, i, j, v):

		if j == sys.maxint:
			j = len(self.seq)
			
		self.seq[i:j] = v

		# how many extra or fewer items?
		d = len(v) - (j-i)

		if d > 0:
			if j > i:
				self.change ((self.HINT_CHANGE, i, j))
			self.change ((self.HINT_INSERT, j, d))
		elif d < 0:
			self.change ((self.HINT_CHANGE, i, j-d))
			self.change ((self.HINT_REMOVE, j-d, d))
		else:
			self.change ((self.HINT_CHANGE, i, j))

	def __delitem__ (self, i):
		del self.seq[i]
		self.change ((self.HINT_REMOVE, i, 1))

	def append (self, v):
		j = len(self.seq)
		self.seq.append (v)
		self.change ((self.HINT_INSERT, j, 1))

	def remove (self, v):
		i = self.seq.index (v)
		self.seq.remove (v)
		self.change ((self.HINT_REMOVE, i, 1))

	def insert (self, i, v):
		self.seq.insert (i, v)
		self.change ((self.HINT_INSERT, i, 1))

	def reverse (self):
		self.seq.reverse()
		self.change ((self.HINT_CHANGE, 0, len(self.seq)))

	def sort (self):
		self.seq.sort()
		self.change ((self.HINT_CHANGE, 0, len(self.seq)))

	# TODO: __add__/__radd__

# An interesting observation: The GDI is really managing views of an
# underlying 'bitmap' model.  When collecting damaged regions, it is
# actually using a sophisticated 'notify' scheme, that is capable of
# collapsing multiple related notifications (regions) into a single
# one.  It receives multiple 'change' messages before sending a
# notify.

class view:

	model = None

	def set_model (self, model):
		self.model = model
		model.add_view (self)	# XXX circular!

	def get_model (self):
		return self.model

class printing_view (view):

	def notify (self, model, hint):
		print 'notify: ', model, 'hint =', hint


if __name__ == '__main__':
	# demonstrate an 'automatic' model object.

	class toggle_button (auto_model):
		def __init__ (self):
			auto_model.__init__ (self, toggle=0, enabled=1)

	bm = toggle_button()
	pv = printing_view()
	bm.add_view (pv)
	print 'toggling button state'
	bm.toggle = not bm.toggle
	bm.enabled = 0
	bm.enabled = 1
