# -*- Mode: Python; tab-width: 4 -*-

# simple button windows

# TODO: lots of expected behavior is still not implemented.  Some
# buttons apparently have a 'repeat' capability (like scroll bar
# buttons).  Also, a button should not 'fire' unless the BUTTONUP
# event happens inside the button.  With the button held down, moving
# into and out of the button should toggle the state.

# TODO: Of course, we need to be able to implement keyboard shortcuts.
# Luckily, since _everything_ uses these button models (toolbars, menus,
# etc...) we should be able to centralize the management of keyboard
# input.  I'd hate to imagine doing this without MVC.

import windc
import winfont
import wingdi
import winwin

import mvc

class push_button_model (mvc.auto_model):
	"push-button model.  pressing the button does not affect the value of toggle"

	# We support lots of attributes.  At first, this seems like
	# overkill, but in fact the push-button model is useful for a wide
	# variety of UI gadgets, including things like button bars, menu
	# bars, menu items, etc...

	def __init__ (self, label=None, pressed=0, toggled=0, armed=0, enabled=1, cascade=0):
		mvc.auto_model.__init__ (
			self,
			pressed	= pressed,	# transient pressed state
			toggled	= toggled,	# state. toggled by pressing.
			armed	= armed,	# ready for action (e.g. mouse is over me)
			enabled	= enabled,
			label	= label,
			cascade	= cascade	# do we kick off other choices? (menus, buttons, etc...)
			)

	def set_pressed_hook (self, bool):
		# can't press it if it's disabled.
		if self.enabled:
			return bool
		else:
			return self.pressed

class toggle_button_model (push_button_model):
	"toggle-button model.  push the button, the toggle toggles"

	def set_pressed_hook (self, bool):
		bool = push_button_model.set_pressed_hook (self, bool)
		if bool:
			self.toggled = not self.toggled
		return bool

user32 = windc.user32

SPLIT_WORD = winwin.SPLIT_WORD

class button (winwin.python_window, mvc.view):

	style = winwin.WS_CHILD | winwin.WS_VISIBLE
	state = 0
	callback = None

	def notify (self, model, hint):
		if model == self.model:
			self.invalidate_rect(0)

	def get_preferred_size (self):
		return 50, 50

	def WM_SIZE (self, wparam, lparam):
		self.size = SPLIT_WORD (lparam)
		self.invalidate_rect (0)

	def WM_DESTROY (self, wparam, lparam):
		self.model.remove_view (self)
		winwin.python_window.WM_DESTROY (self, wparam, lparam)

	def WM_PAINT (self, wparam, lparam):
		dc = self.begin_paint()
		self.prepare_dc (dc)
		self.do_paint (dc)
		self.end_paint()

	def do_paint (self, dc):
		m = self.model
		rect = self.get_client_rect()
		down = m.pressed or m.toggled

		if m.pressed:
			kind = 'sunken'
		elif m.armed:
			kind = 'raised'
		else:
			kind = 'etched'

		dc.draw_3d_box (rect, kind)

	def prepare_dc (self, dc):
		pass

	def get_dc (self):
		dc = winwin.python_window.get_dc (self)
		self.prepare_dc (dc)
		return dc

	# 3 states:
	OUTSIDE	= 0
	INSIDE	= 1
	PRESSED	= 2

	capture_state = 0

	def WM_MOUSEMOVE (self, wparam, lparam):
		mx, my = SPLIT_WORD (lparam)

		if self.capture_state == self.OUTSIDE:
			self.set_capture()
			self.capture_state = self.INSIDE
			self.model.armed = 1
		elif self.capture_state == self.PRESSED:
			# we don't care where the mouse moves
			pass
		else:
			l,t,r,b = self.get_client_rect()
			if not ((l <= mx < r) and (t <= my < b)):
				self.release_capture()
				self.capture_state = self.OUTSIDE
				self.model.armed = 0
			else:
				self.model.armed = 1

	def WM_LBUTTONDOWN (self, wparam, lparam):
		mx, my = SPLIT_WORD (lparam)
		# can only happen if capture_state == INSIDE
		self.model.pressed = 1
		self.capture_state = self.PRESSED
		
	def WM_LBUTTONUP (self, wparam, lparam):
		mx, my = SPLIT_WORD (lparam)
		l,t,r,b = self.get_client_rect()

		self.model.pressed = 0

		if not ((l <= mx < r) and (t <= my < b)):
			self.release_capture()
			self.capture_state = self.OUTSIDE
			self.model.armed = 0
		else:
			self.capture_state = self.INSIDE

class text_button (button):
	
	font = wingdi.get_stock_object (wingdi.SYSTEM_FONT)

	text_color		= windc.get_sys_color (windc.COLOR_MENUTEXT)
	hl_color		= windc.get_sys_color (windc.COLOR_3DHILIGHT)
	shadow_color	= windc.get_sys_color (windc.COLOR_3DSHADOW)

	# Need to catch a label change.

	def get_preferred_size (self):
		m = self.model
		if m.label is not None:
			dc = self.get_dc()
			dc.select_object (self.font)
			w, h = dc.get_text_extent (m.label)
			self.label_size = w, h
			return (w+2*h), (3*h)/2
		else:
			return 50, 50

	def prepare_dc (self, dc):
		dc.set_bk_color()
		dc.set_bk_mode ()
		dc.select_object (self.font)

	def do_paint (self, dc):
		m = self.model
		button.do_paint (self, dc)
		l,t,r,b = self.get_client_rect()
		h = b-t
		w = r-l
		tw, th = self.label_size
		x, y = l+((w-tw)/2), t+((h-th)/2)
		if m.enabled:
			dc.set_text_color (self.text_color)
			if m.armed:
				x, y = x+1, y+1
			if m.toggled:
				x, y = x-1, y-1
			dc.text_out ((x,y), m.label)
		else:
			dc.set_text_color (self.hl_color)
			dc.text_out ((x+1,y+1), m.label)
			dc.set_text_color (self.shadow_color)			
			dc.text_out ((x,y), m.label)

def make_text_button (label, parent):
	m = push_button_model (label=label)
	b = text_button (parent=parent)
	b.set_model (m)
	return b

def test():
	import layout

	parent = layout.flow_container ('Button Demo', w=200, h=100).create()
	pv = mvc.printing_view()

	button_models = []
	for label in ('one', 'two', 'three', 'four'):
		tbm = toggle_button_model (label=label)
		child = text_button ('text button %s' % label)
		child.set_model (tbm)
		child.parent = parent
		parent.add (child)
		child.create()
		child.show_window()
		tbm.add_view (pv)
		button_models.append (tbm)

	class enable_toggle (toggle_button_model):

		def set_toggled_hook (self, bool):
			if bool != self.toggled:
				if bool:
					self.label = 'enable buttons'
				else:
					self.label = 'disable buttons'

				for sm in self.other_models:
					sm.enabled = not sm.enabled

			return bool

	etm = enable_toggle (label='disable buttons')
	etm.new_property ('other_models', button_models)
	child = text_button ()
	child.set_model (etm)
	child.parent = parent
	parent.add (child, None)
	child.create()

	parent.show_window()
	return etm

if __name__ == '__main__':
	import msgloop
	test()
	msgloop.go()
