# -*- Mode: Python; tab-width: 4 -*-

#
# A button bar component
#

# IDEA: there's usually a lot of wasted space to the right of the menu
# bar, why not use _that_ space for a status line? (using flow layout?)

import windc
import winfont
import wingdi
import winwin
user32 = windc.user32
SPLIT_WORD = winwin.SPLIT_WORD

import mvc
import mvc_button
import mvc_menu

# uses a list of button models to present a button/menu bar.

class button_bar_component (winwin.python_window, mvc.view):

	style = winwin.WS_CHILD | winwin.WS_VISIBLE
	font = winfont.get_system_fonts()['menu_font'].create()
	buttons = None
	border_ratio = (3, 2)

	font_height = 0

	def get_font_height (self):
		# memoized
		if self.font_height == 0:
			dc = winwin.python_window.get_dc (self)
			dc.select_object (self.font)
			tm = dc.get_text_metrics()
			dc.release_dc()
			self.font_height = tm.height

		return self.font_height

	def get_preferred_size (self):
		n, d = self.border_ratio
		w = reduce (lambda a,b: a+b, self.widths, 0)
		return w, (n * self.get_font_height()) / d

	def create (*args):
		self = apply (winwin.python_window.create, args)
		for button in self.buttons:
			button.add_view (self)
		self.recalc()
		return self

	def WM_DESTROY (self, wparam, lparam):
		for b in self.buttons:
			b.remove_view (self)
		return winwin.python_window.WM_DESTROY (self, wparam, lparam)

	def recalc (self):
		dc = self.get_dc()
		dc.select_object (self.font)
		widths = []
		for button in self.buttons:
			w,h = dc.get_text_extent (button.label)
			widths.append (h + w)

		self.widths = widths

		dc.release_dc()

	def WM_SIZE (self, wparam, lparam):
		self.size = SPLIT_WORD (lparam)
		self.invalidate_rect()
		return 0

	def WM_ERASEBKGND (self, wparam, lparam):
		dc = windc.DC (wparam)
		dc.fill_rect (dc.get_clip_box(), windc.get_sys_brush (windc.COLOR_3DFACE))
		return 1

	def WM_PAINT (self, wparam, lparam):
		dc = self.begin_paint()
		self.prepare_dc (dc)
		for i in range(len(self.buttons)):
			self.draw_button (dc, i)

		self.end_paint()

	def prepare_dc (self, dc):
		dc.set_bk_color()
		dc.set_bk_mode ()
		dc.select_object (self.font)
		dc.set_text_align (wingdi.TA_CENTER)

	def get_dc (self):
		dc = winwin.python_window.get_dc (self)
		self.prepare_dc (dc)
		return dc

	def get_button_rect (self, i):
		x = 0
		for j in range(i+1):
			bw = self.widths[j]
			x = x + bw
		x = x - bw
		w, h = self.size
		return (x, 0, x+bw, h)

	def draw_button (self, dc, i):
		rect = self.get_button_rect (i)
		button = self.buttons[i]

		if button.pressed or button.toggled:
			kind = 'sunken'
		elif button.armed:
			kind = 'raised'
		else:
			kind = 'etched'

		dc.draw_3d_box (rect, kind)

		l,t,r,b = rect
		fh = self.get_font_height()
		bh = b-t
		x = l + ((r-l)/2)
		y = t + ((bh-fh)/2)

		if button.pressed:
			x, y = x + 1, y + 1

		dc.text_out ((x, y), button.label)
		
	captured = None

	def WM_LBUTTONDOWN (self, wparam, lparam):
		mx, my = SPLIT_WORD (lparam)
		# did they hit a button?
		for i in range(len(self.buttons)):
			l,t,r,b = self.get_button_rect (i)
			if l < mx < r:
				self.buttons[i].pressed = 1
				self.set_capture()
				self.captured = i

	def WM_LBUTTONUP (self, wparam, lparam):
		if self.captured is not None:
			self.buttons[captured].pressed = 0
			self.release_capture()
			self.captured = None

class menu_bar_component (button_bar_component):


	def WM_LBUTTONDOWN (self, wparam, lparam):
		mx, my = SPLIT_WORD (lparam)
		# did they hit a button?
		for i in range(len(self.buttons)):
			l,t,r,b = self.get_button_rect (i)
			if l < mx < r:
				self.buttons[i].pressed = 1

	def WM_LBUTTONUP (self, wparam, lparam):
		pass

	armed_model = None
	captured = 0

	def WM_MOUSEMOVE (self, wparam, lparam):
		mx, my = SPLIT_WORD (lparam)

		# we capture the mouse to implement
		# 'eagerly armed' behavior..

		if not self.captured:
			self.mouse_enter()
		else:
			l,t,r,b = self.get_client_rect()
			if not ((l <= mx < r) and (t <= my < b)):
				# outside the window
				self.mouse_leave()
			else:
				for i in range(len(self.buttons)):
					l,t,r,b = self.get_button_rect (i)
					bm = self.buttons[i]
					if (l < mx < r) and (t < my < b):
						bm.armed = 1
						return

	def mouse_enter (self):
		self.set_capture()
		self.captured = 1

	def mouse_leave (self):
		self.release_capture()
		self.captured = 0
		if self.armed_model is not None:
			if self.armed_model.pressed:
				pass
			else:
				self.armed_model.armed = 0
				self.armed_model = None

	def notify (self, model, hint):
		# search for the model...
		i = 0
		for button in self.buttons:
			if button is model:
				self.notify_model (i, model, hint)
				break
			i = i + 1

	def notify_model (self, i, model, hint):
		# hook the behavior of armed/pressed

		if hint == 'armed':
			if model.armed:
				# make sure that only one model at a time is armed.
				if self.armed_model is not None:
					self.armed_model.armed = 0
				self.armed_model = model

		# draw the changes...
		dc = self.get_dc()
		self.draw_button (dc, i)
		dc.release_dc()

# See mvc_model_tree.py to test.
