# -*- Mode: Python; tab-width: 4 -*-

# simple single-line edit control, uses a sequence model for the
# string of characters.

import mvc
import string
import winfont
import winmesg
import winwin
user32 = winwin.user32

SPLIT_WORD = winwin.SPLIT_WORD

class edit_control (winwin.python_window):

	line_height = 14
	font = winfont.font ('Arial', height=line_height).create()

	def __init__ (self, parent, text='', width=15, observer=None):
		m = self.model = mvc.sequence_model (map (None, text))
		self.width = width
		self.cursor_pos = len(m)
		self.scroll_pos = 0
		self.observer = observer
		m.add_view (self)
		winwin.python_window.__init__ (
			self,
			parent=parent,
			style=winwin.WS_CHILD
			)

	char_widths = None

	def get_text (self):
		return string.join (self.model, '')

	def get_char_width (self, ch):
		if self.char_widths is None:
			dc = self.get_dc()
			dc.select_object (self.font)
			self.char_widths = dc.get_char_width (0,127)
			dc.release_dc()
		return self.char_widths[ord(ch)]

	# this default helps set_caret_pos during window creation.
	size = 100, line_height+4

	def WM_SIZE (self, wparam, lparam):
		self.size = SPLIT_WORD (lparam)

	def WM_PAINT (self, wparam, lparam):
		dc = self.begin_paint()
		dc.set_bk_color()
		dc.set_bk_mode()
		self.do_paint (dc)
		self.end_paint()

	def notify (self, model, hint):
		kind = hint[0]
		if kind == model.HINT_CHANGE:
			ignore, low, high = hint
			start = low
		elif kind == model.HINT_INSERT:
			ignore, start, nitems = hint
		elif kind == model.HINT_REMOVE:
			ignore, start, nitems = hint
		# redraw from <start> to the end.
		x = self.get_char_pos (start)
		l,t,r,b = self.get_client_rect()
		# don't let the inset flicker...
		self.invalidate_rect ((x,t+2,r-2,b-2))

	def do_paint (self, dc):
		r = self.get_client_rect()
		dc.draw_3d_box (r, kind='sunken', filled=0)
		dc.select_object (self.font)
		sx = self.scroll_pos
		dc.text_out ((3,2), self.get_text()[sx:])

	def get_preferred_size (self):
		w = self.line_height * self.width
		h = self.line_height
		# need extra room for border
		return w+4, h+4
	
	def WM_SETFOCUS (self, wparam, lparam):
		self.show_caret()
	
	def WM_KEYDOWN (self, wparam, lparam):
		cp = self.cursor_pos
		if wparam == winmesg.VK_LEFT:
			cp = cp - 1
		elif wparam == winmesg.VK_RIGHT:
			cp = cp + 1
		if (cp < 0) or (cp > len(self.model)):
			winwin.user32.MessageBeep(0)
		else:
			self.cursor_pos = cp
			self.set_caret_pos()

	# interesting idea
	def WM_MOUSEMOVE (self, wparam, lparam):
		self.set_focus()

	def WM_CHAR (self, wparam, lparam):
		ch = chr (wparam)

		if ch == '\010':
			# backspace
			if self.cursor_pos > 0:
				del self.model[self.cursor_pos-1]
				self.cursor_pos = self.cursor_pos - 1
				self.set_caret_pos()
			else:
				winwin.user32.MessageBeep (0)
		# notifications
		elif ch in '\r\n':
			if self.observer:
				self.observer.notify (self.model, 'enter')
		elif ch == '\t':
			if self.observer:
				self.observer.notify (self.model, 'tab')				
		else:
			x = self.cursor_pos
			self.model.insert (x, ch)
			self.cursor_pos = self.cursor_pos + 1
			self.set_caret_pos()

	def show_caret (self):
		user32.CreateCaret (self, 0, 2, self.line_height-2)
		self.set_caret_pos()
		user32.ShowCaret (self)

	def get_char_pos (self, index):
		sx = self.scroll_pos
		return reduce (
			lambda a,b: a+b,
			map (
				self.get_char_width,
				self.model[sx:index]
				),
			0
			)

	def set_caret_pos (self):
		w, h = self.size
		x = self.get_char_pos (self.cursor_pos)
		if x >= (w-4):
			self.scroll_pos = self.scroll_pos + 1
			(l,t,r,b) = self.get_client_rect()
			self.invalidate_rect((l+2,t+2,r-2,b-2))
			self.set_caret_pos()
		elif (x == 0) and self.scroll_pos:
 			self.scroll_pos = max (0, self.scroll_pos - 10)
 			(l,t,r,b) = self.get_client_rect()
 			self.invalidate_rect((l+2,t+2,r-2,b-2))
 			self.set_caret_pos()
		else:
			user32.SetCaretPos (x+2, 3)

	def WM_KILLFOCUS (self, wparam, lparam):
		user32.DestroyCaret (self)

import layout
import static

def test():

	class edit_test (layout.vbox):
		
		def WM_SETFOCUS (self, wparam, lparam):
			self.edit_control.set_focus()
			
		def create (*args):
			self = apply (layout.vbox.create, args)
			e = self.edit_control = edit_control (self, 'Howdy There').create()
			s = static.static_text_component (self, 'Edit Test').create()
			self.add (self.GLUE)
			self.add (s)
			self.add (self.spacer (10, 10))
			self.add (e)
			self.add (self.GLUE)
			for w in self,e,s:
				w.show_window()

			return self

	return edit_test (
		style=winwin.WS_OVERLAPPEDWINDOW,
		w=300, h=100
		).create()
 
