# -*- Mode: Python; tab-width: 4 -*-

import windc
import winfont
import wingdi
import winwin
import string

import mvc_button

SPLIT_WORD = winwin.SPLIT_WORD

# FIXME: The whole thing needs to be redone, with everything centered
# around the _baseline_ of the text, not the bounding rect.  It's WAY
# overcomplicated.

# TODO: I have noticed that on win32 menus, there is a slight delay
# (~.75 seconds?) before a cascaded menu pops up (or goes away).
# [this is documented in the user-interface guidelines]

# TODO: menu captions would be nice, though it can of course be done
# by using a window class with a title bar... [see <caption_item>
# below, a start...]

class item:
	menu_color		= windc.get_sys_color	(windc.COLOR_MENU)
	menu_brush		= windc.get_sys_brush	(windc.COLOR_MENU)
	menu_pen		= windc.get_sys_pen		(windc.COLOR_MENU)
	text_color		= windc.get_sys_color	(windc.COLOR_MENUTEXT)
	text_pen		= windc.get_sys_pen		(windc.COLOR_MENUTEXT)
	text_brush		= windc.get_sys_brush	(windc.COLOR_MENUTEXT)
	hl_text_color	= windc.get_sys_color	(windc.COLOR_HIGHLIGHTTEXT)
	hl_color		= windc.get_sys_color	(windc.COLOR_3DHILIGHT)
	shadow_color	= windc.get_sys_color	(windc.COLOR_3DSHADOW)
	shadow_pen		= windc.get_sys_pen		(windc.COLOR_3DSHADOW)
	shadow_brush	= windc.get_sys_brush	(windc.COLOR_3DSHADOW)
	hl_pen			= windc.get_sys_pen		(windc.COLOR_3DHILIGHT)
	hl_brush		= windc.get_sys_brush	(windc.COLOR_3DHILIGHT)

class text_item (item):

	"menu item based on a push_button_model"

	def __init__ (self, model, **props):
		if type(model) == type(''):
			# convenient
			self.model = mvc_button.push_button_model (label=model)
		else:
			self.model = model

		for k,v in props.items():
			setattr (self.model, k, v)

	def draw (self, dc, y, w, mh, mw, fh):
		# base offset
		bmh = (mh - fh)/2
		m = self.model
		fh2 = fh/2
		fh4 = fh/4

		cascade_rect = ((w-4)-fh, y-fh, (w-4)-fh2, y-fh4)

		if not m.armed:
			dc.select_object (self.menu_pen)
			dc.select_object (self.menu_brush)					
			dc.rectangle ((4,y-mh+bmh,w-4,y+bmh))

		if m.enabled:
			if m.armed:
				dc.select_object (self.shadow_brush)
				dc.select_object (self.shadow_pen)
				dc.rectangle ((4,y-mh+bmh,w-4,y+bmh))
				dc.select_object (self.hl_pen)
				dc.select_object (self.hl_brush)
				dc.set_text_color (self.hl_text_color)
			else:
				dc.select_object (self.text_pen)
				dc.select_object (self.text_brush)
				dc.set_text_color (self.text_color)

			self.draw_cascade_indicator (dc, cascade_rect)
			dc.text_out ((mw, y), m.label)
		else:
			# disabled state
			dc.set_text_color (self.hl_color)
			dc.text_out ((mw+1, y+1), m.label)
			dc.set_text_color (self.shadow_color)
			dc.select_object (self.shadow_pen)
			dc.select_object (self.shadow_brush)
			self.draw_cascade_indicator (dc, cascade_rect)
			dc.text_out ((mw, y), m.label)

	def draw_cascade_indicator (self, dc, rect):
		if self.model.cascade:
			# draw a triangle
			l,t,r,b = rect
			dc.select_object (self.text_pen)
			#dc.rectangle (rect)
			dc.polygon ((l,t), (r,t+((b-t)/2)), (l,b))

class separator_item (item):

	def draw (self, dc, y, w, mh, mw, fh):
		# base offset
		bmh = (mh - fh)/2
		my = y-(mh/2)
		dc.select_object (self.shadow_pen)
		dc.move_to ((4, my))
		dc.line_to ((w-4, my))
		dc.select_object (self.hl_pen)
		dc.move_to ((4, my+1))
		dc.line_to ((w-4, my+1))

class caption_item (item):

	def __init__ (self, caption):
		self.caption = caption

	def draw (self, dc, y, w, mh, mw, fh):
		dc.text_out ((mw, y), self.caption)

# We really only need one of these
separator = separator_item()

class menu (winwin.python_window):
	
	non_client_metrics = winfont.get_non_client_metrics()
	font = winfont.get_system_fonts(non_client_metrics)['menu_font'].create()
	font_height = abs (font.lf.height)
	# height of a menu _bar_
	menu_height = non_client_metrics.menu_height
	# width of menu-bar buttons
	menu_width  = non_client_metrics.menu_width

	style = winwin.WS_POPUP

	menu_color		= windc.get_sys_color (windc.COLOR_MENU)
	menu_brush		= windc.get_sys_brush (windc.COLOR_MENU)
	menu_pen		= windc.get_sys_pen   (windc.COLOR_MENU)
	text_color		= windc.get_sys_color (windc.COLOR_MENUTEXT)
	hl_text_color	= windc.get_sys_color (windc.COLOR_HIGHLIGHTTEXT)
	hl_color		= windc.get_sys_color (windc.COLOR_3DHILIGHT)
	shadow_color	= windc.get_sys_color (windc.COLOR_3DSHADOW)
	shadow_pen		= windc.get_sys_pen   (windc.COLOR_3DSHADOW)
	shadow_brush	= windc.get_sys_brush (windc.COLOR_3DSHADOW)
	hl_pen			= windc.get_sys_pen   (windc.COLOR_3DHILIGHT)

	def create (*args):
		# I know this isn't right - but it's pretty much impossible to
		# figure out what style windows uses, because the menu goes
		# away on KILLFOCUS.
		args[0].style = winwin.WS_POPUP | winwin.WS_VISIBLE
		args[0].ext_style = winwin.WS_EX_TOOLWINDOW

		self = apply (winwin.python_window.create, args)

		# calculate the size of the menu
		dc = self.get_dc()
		
		dc.select_object (self.font)
		widest = 0
		for item in self.items:
			if hasattr (item, 'model'):
				m = item.model
				m.add_view (self)
				w, h = dc.get_text_extent (m.label)
				widest = max (widest, w)

		dc.release_dc()
		mh = self.menu_height
		fh = self.font_height
		bmh = (mh - fh)/2
		self.move_window (
			self.x,
			self.y,
			widest + self.menu_width + (fh*2),
			(mh * len(self.items)) + bmh + 4
			)
		return self

	def WM_CHAR (self, wparam, lparam):
		# TODO: real keyboard control...
		ch = chr (wparam)
		if ch == '\033':
			# escape
			self.destroy_window()

	def WM_DESTROY (self, wparam, lparam):
		for x in self.items:
			if hasattr (x, 'model'):
				x.model.remove_view (self)
		return winwin.python_window.WM_DESTROY (self, wparam, lparam)

	def WM_KILLFOCUS (self, wparam, lparam):
		self.destroy_window()

	def WM_PAINT (self, wparam, lparam):
		dc = self.begin_paint()
		self.draw_menu (dc)
		self.end_paint()

	# a tuple of (line_number, model).
	#
	# { should be re-written to use the invariant/reactive
	#   technique instead of a member variable }
	highlighted = None, None

	def WM_MOUSEMOVE (self, wparam, lparam):
		x, y =  SPLIT_WORD (lparam)
			
		index = y / self.menu_height
		dc = self.get_dc()
		self.prepare_dc (dc)

		hl, hm = self.highlighted

		if (index < len(self.items)) and (index >= 0):
			item = self.items[index]
			if hasattr (item, 'model'):
				m = item.model
			else:
				m = None

			if hl != index:
				if hl is not None:
					hm.armed = 0
				if m:
					m.armed = 1
					self.highlighted = index, m
		else:
			if hl is not None:
				hm.armed = 0
				self.highlighted = None, None

		dc.release_dc()

	def WM_LBUTTONDOWN (self, wparam, lparam):
		x, y = SPLIT_WORD (lparam)
		index = y / self.menu_height

		if (index < len(self.items)) and (index >= 0):
			item = self.items[index]
			if hasattr (item, 'model'):
				item.model.pressed = 1
				self.set_capture()
				self.captured = item
				self.destroy_window()

	captured = None

	def WM_LBUTTONUP (self, wparam, lparam):
		self.release_capture()
		if self.captured:
			self.captured.model.pressed = 0
			self.captured = None

	def prepare_dc (self, dc):
		dc.select_object (self.font)
		dc.set_bk_color()
		dc.set_bk_mode (windc.TRANSPARENT)
		dc.set_text_align (wingdi.TA_BOTTOM)

	def draw_line (self, dc, index):
		mw = self.menu_width
		dy = self.menu_height
		fh = self.font_height
		l,t,r,b = self.get_client_rect()
		w, h = r-l, b-t
		y = (index+1) * dy

		object = self.items[index]
		object.draw (dc, y, w, dy, mw, fh)

	def draw_menu (self, dc):

		self.prepare_dc (dc)
		dc.draw_3d_box (self.get_client_rect())

		for i in range (len (self.items)):
			self.draw_line (dc, i)

	def notify_model (self, i, model, hint):
		dc = self.get_dc()
		self.prepare_dc (dc)
		self.draw_line (dc, i)
		dc.release_dc()

	def notify (self, model, hint):
		# search for the model...
		i = 0
		for item in self.items:
			if hasattr (item, 'model'):
				if item.model is model:
					self.notify_model (i, model, hint)
					break
			i = i + 1

def test():
	import mvc
	# show off the power of MVC, by tying two menus
	# to the same set of push-button models.
	m1 = menu(x=100,y=100)
	m2 = menu(x=500,y=100)

	pbm = mvc_button.push_button_model
	pv = mvc.printing_view()
	
	items = [
		text_item (pbm (label='Item One')),
		text_item (pbm (label='Item Two')),
		separator,
		text_item (pbm (label='Collapse Universe')),
		text_item (pbm (label='Whirled Peas')),
		separator,
		text_item (pbm (label='Disabled Item', enabled=0)),
		text_item (pbm (label='Enabled Item')),
		text_item (pbm (label='Crash Windows 95')),
		]
	
	for item in items:
		if item is not separator:
			item.model.add_view (pv)

	m1.items = m2.items = items
	m1.create()
	m2.create()
	l,t,r,b = m2.get_window_rect()
	m1.show_window()
	m2.show_window()
	return m1, m2

if __name__ == '__main__':
	import msgloop
	w1, w2 = test()
	msgloop.go()
