# -*- Mode: Python; tab-width: 4 -*-

import mvc_button
import mvc_button_bar
import mvc_menu

# ===========================================================================
#							  Model Tree
# ===========================================================================

# A model tree is a list of (push-button) models and other model trees.
# [ <parent> <child> <child> <child> ]
# Where <child> can be either a single model,
# or another menu tree.
#
# [ <parent_1>
#       <child_1>
#       [ <parent_2>
#             <child_3>
#             <child_4> ]
#       <child_5> ]
#
# The children of <parent_1> are (<child_1>, <parent_2>, and <child_5>)
# 
# [todo: a None in the list ought to represent a 'separator']
#
# This is a very general model, and opens up lots of possibilities for
# user-configurable interfaces.  For example, imagine a menu bar with
# a little 'back' button, that lets you crawl up the menu hierarchy
# almost like a browser.  Or tear-off menus that turn into toolbars.

class model_tree:

	"""A model tree is used to build hierachical option 'trees', which can
	be represented as tree of cascaded menus, with the top level usually
	implemented as a button bar."""

	def __init__ (self, tree=None):
		self.tree = tree

	def get_child_models (self):

		def flatten (thing):
			if type(thing) is type([]):
				model = thing[0]
				# make sure this model knows it's a cascade
				model.cascade = 1
			else:
				model = thing
			return model

		return map (flatten, self.tree[1:])

	def has_children (self, model):
		for x in self.tree[1:]:
			if type(x) is type([]):
				if x[0] is model:
					return 1
		return 0

	def get_child_tree (self, model):
		for x in self.tree[1:]:
			if type(x) is type([]):
				if x[0] is model:
					return model_tree (x)

	def as_menu_bar (self, **props):
		"Create a menu bar based on this model tree"

		mb = menu_bar_component (window_name = self.tree[0].label)

		for k,v in props.items():
			setattr (mb,k,v)

		mb.buttons = self.get_child_models()
		mb.model_tree = self # cycle
		return mb

	def as_popup_menu (self, **props):
		"Create a popup menu based on this model tree"
		
		pm = menu_for_model_tree (window_name = self.tree[0].label)
		pm.items = map (mvc_menu.text_item, self.get_child_models())

		for k,v in props.items():
			setattr (pm,k,v)

		pm.model_tree = self # cycle
		return pm
		
	def add_view_all (self, view):
		map_tree (
			lambda x,v=view: x.add_view (v),
			self.tree
			)

	def remove_view_all (self, view):
		map_tree (
			lambda x,v=view: x.remove_view (v),
			self.tree
			)

def map_tree (function, tree):
	"map a function over a tree"
	if type(tree) is type([]):
		return map (
			lambda x,f=function: map_tree (f, x),
			tree
			)
	else:
		return function (tree)

def make_model_tree (label_tree):
	"build a model tree out of a label tree"

	def make_model (label):
		return mvc_button.push_button_model (label=label)

	return model_tree (map_tree (make_model, label_tree))


# The tricky part in handling cascaded menus and menu bars is in
# maintaining the 'only-one-child' invariant.  Eventually we will
# have to deal completely with the issue of keyboard focus, too.

# ===========================================================================
#					  Menu Bar for a Model Tree
# ===========================================================================

class menu_bar_component (mvc_button_bar.menu_bar_component):

	def notify_model (self, index, model, hint):
		mvc_button_bar.menu_bar_component.notify_model (self, index, model, hint)
		if model.armed:
			if model.cascade:
				if model.pressed:
					self.create_child (index, model)
		elif self.is_child_model (model):
			# if the child model becomes unarmed, then detach it.
			self.detach_child()

	child_menu = None
	parent_menu = None

	def create_child (self, index, model):
		# detach any other child
		self.detach_child()

		# create the child menu
		ct = self.model_tree.get_child_tree (model)
		l,t,r,b = self.get_window_rect()
		br = self.get_button_rect (index)
		mw = ct.as_popup_menu (x = l+br[0], y = b-1)
		mw.create()
		mw.show_window()

		# build some (cyclic) links.
		self.child_menu = mw
		mw.parent_menu = self
		mw.model = model

	def is_child_model (self, model):
		return self.child_menu and (self.child_menu.model is model)

	def detach_child (self):
		if self.child_menu is not None:
			cm = self.child_menu
			cm.detach_child()
			cm.destroy_window()
			cm.model.armed = 0
			cm.model.pressed = 0
			self.child_menu = None

	def WM_KILLFOCUS (self, wparam, lparam):
		if (self.child_menu is None) or (self.child_menu.hwnd == wparam):
			return
		else:
			self.detach_child()

# ===========================================================================
#						Menu for a Model Tree
# ===========================================================================

# Scanning the KB (Q84190), I see that POPUP (i.e., non-child) windows
# can have a 'parent' and/or 'owner'.  Not sure what the distinction
# is, but using it may handle the destruction of 'dependent' popups
# for us?

class menu_for_model_tree (mvc_menu.menu):

	def WM_DESTROY (self, wparam, lparam):
		hl, hm = self.highlighted
		if hm is not None:
			hm.armed = 0
		return mvc_menu.menu.WM_DESTROY (self, wparam, lparam)

	def notify_model (self, index, model, hint):
		mvc_menu.menu.notify_model (self, index, model, hint)
		if model.armed:
			if model.cascade:
				self.create_child (index, model)
			else:
				self.detach_child()

	child_menu = None
	parent_menu = None

	def create_child (self, index, model):
		# detach any other child first...
		self.detach_child()

		mt = self.model_tree.get_child_tree (model)
		l,t,r,b = self.get_window_rect()
		mw = mt.as_popup_menu (x=r-3, y=(self.menu_height * index) + t)

		# build some (cyclic) links.
		self.child_menu = mw
		mw.parent_menu = self
		mw.model = model

		mw.create()
		mw.show_window()

	def detach_child (self):
		if self.child_menu is not None:
			self.child_menu.detach_child()
			self.child_menu.destroy_window()
			self.child_menu = None

	def WM_KILLFOCUS (self, wparam, lparam):

		if self.child_menu is None:
			# if the tail menu loses focus, find the top-most
			# ancestor and kill the whole line.
			m = self
			while m.parent_menu:
				if m.parent_menu.hwnd == wparam:
					# unless the focus is moving to a parent.
					return
				else:
					m = m.parent_menu
			m.detach_child()

def test():
	sample_tree = [
		'Top-Level Menu',
		['File',
		 'Open',
		 'Close',
		 'Yorble',
		 ['Destroy',
		  'Maliciously',
		  'Cavalierly',
		  'With Extreme Prejudice'
		  ],
		 'Mangle'
		 ],
		['Print',
		 'Setup',
		 'Fax',
		 'To USPS'
		 ],
		['Contemplate',
		 'Beta-Expansion of the Applicative-Order Y Combinator',
		 'Quantum Chromodynamics',
		 'Observer-Created Reality',
		 ],
		['LRU',
		 ['Bus',
		  'Novel',
		  'Paragraph',
		  'Sentence',
		  ['Participle',
		   'Dangling',
		   'Fraternizing'],
		  'Verb Phrase',
		  ['Word',
		   'Analog',
		   'Discrete',
		   ['Binary',
			'Float',
			'Ratio'],
		   ]],
		 'Power/Ground',
		 'SSM Type'
		 ]
		]

	import winwin
	mt =  make_model_tree (sample_tree)
	mb = mt.as_menu_bar(
		style = winwin.WS_OVERLAPPEDWINDOW,
		ext_style = winwin.WS_EX_TOOLWINDOW,
		h = 60, w = 640
		).create()
	mb.show_window()

if __name__ == '__main__':
	import msgloop
	test()
	msgloop.go()
