# -*- Mode: Python; tab-width: 4 -*-

import layout
import windc
import winwin
import mvc
import mvc_button

import sys

SPLIT_WORD = winwin.SPLIT_WORD

# ===========================================================================
# Scroll Bar/Slider
# ===========================================================================

class scroll_model (mvc.auto_model):

	def __init__ (self, position=0, page=100, limit=100):
		mvc.auto_model.__init__ (self, position=position, page=page, limit=limit)

	def set_position_hook (self, pos):
		# don't let position go out of range
		if pos < 0:
			return 0
		if (pos + self.page) > self.limit:
			return self.limit - self.page
		return pos

	def set_page_hook (self, page):
		page = min (self.limit, page)
		if (self.position + page) > self.limit:
			# scoot position back if page moves it out of range
			self.position = self.limit - page

		return page

class vertical_scrollbar (winwin.python_window, mvc.view):

	style = winwin.WS_CHILD | winwin.WS_VISIBLE
	width = w = 14

	def notify (self, model, hint):
		if model == self.model:
			self.move_thumb()

	def WM_SIZE (self, wparam, lparam):
		self.invalidate_rect (0)

	def WM_PAINT (self, wparam, lparam):
		dc = self.begin_paint()
		self.do_paint (dc)
		self.end_paint()

	captured = None

	def WM_LBUTTONDOWN (self, wparam, lparam):
		x, y = SPLIT_WORD (lparam)
		tl, tt, tr, tb = self.get_thumb_rect()
		# did they hit the thumb?
		if (y >= tt) and (y < tb):
			self.captured = y - tt
			self.set_capture()
		# TODO: handle 'page' clicks in the background
		# area.
			
	_last_thumb_rect = None

	def WM_MOUSEMOVE (self, wparam, lparam):
		if self.captured is not None:
			x, y = SPLIT_WORD (lparam)
			# adjust y for offset into thumb
			y = y - self.captured

			tl,tt,tr,tb = self.get_thumb_rect()
			cl,ct,cr,cb = self.get_client_rect()

			# client height
			ch = cb - ct
			# thumb height
			th = tb - tt

			# range check
			if y < 0:
				# top
				y = 0
			elif y + th >= cb:
				# bottom (taking page size into account)
				y = cb - th

			# convert position to n/d
			self.model.position = (y * self.model.limit) / ch

	def move_thumb (self):
		thumb_rect = self.get_thumb_rect()
		if thumb_rect != self._last_thumb_rect:
			cl,ct,cr,cb = self.get_client_rect()
			tl,tt,tr,tb = thumb_rect
			dc = self.get_dc()
			self.draw_thumb (dc, self.get_thumb_rect())
			brush = windc.get_sys_brush (windc.COLOR_WINDOW)
			dc.fill_rect ((cl,ct,cr,tt-1), brush)
			dc.fill_rect ((cl,tb+1,cr,cb), brush)
			dc.release_dc()
			self._last_thumb_rect = thumb_rect

	def WM_LBUTTONUP (self, wparam, lparam):
		if self.captured is not None:
			self.captured = None
			self.release_capture()

	minimum_thumb_size = 10

	def get_thumb_rect (self):
		l,t,r,b = self.get_client_rect()
		w = r-l
		h = b-t

		m = self.model
		pos, lim, pag = m.position, m.limit, m.page
		
		if lim == 0:
			tp = 0
			th = h
		else:
			tp = (pos * h) / lim
			th = (pag * h) / lim

		if th > h:
			th = h
		if th < self.minimum_thumb_size:
			th = self.minimum_thumb_size
		ty = t+tp
		return l, ty, r, ty+th

	def draw_thumb (self, dc, rect):
		dc.draw_3d_box (rect, 'etched')
		# add some texture, with a small inset wedge
		l,t,r,b = rect
		offset = 2
		cy = t + ((b-t)/2)
		dc.draw_3d_box ((l+offset, (cy-3), r-offset, (cy+3)), 'etched')

	def do_paint (self, dc):
		back = self.get_client_rect()
		dc.fill_rect (back, windc.get_sys_brush (windc.COLOR_WINDOW))
		self.draw_thumb (dc, self.get_thumb_rect())
	
	def get_preferred_size (self):
		return self.width, sys.maxint

# most of the code has to be rewritten.  maybe some day we will have a
# fully general coordinate-transform based system, where we need only
# specify a 90-degree rotation transform. 8^(

class horizontal_scrollbar (vertical_scrollbar):

	def WM_LBUTTONDOWN (self, wparam, lparam):
		x, y = SPLIT_WORD (lparam)
		tl, tt, tr, tb = self.get_thumb_rect()
		# did they hit the thumb?
		if (x >= tl) and (x < tr):
			self.captured = x - tl
			self.set_capture()

	def WM_MOUSEMOVE (self, wparam, lparam):
		if self.captured is not None:
			x, y = SPLIT_WORD (lparam)
			# adjust x for offset into thumb
			x = x - self.captured

			tl,tt,tr,tb = self.get_thumb_rect()
			cl,ct,cr,cb = self.get_client_rect()

			# client width
			cw = cr - cl
			# thumb width
			tw = tr - tl

			# range check
			if x < 0:
				# top
				x = 0
			elif x + tw >= cw:
				# bottom (taking page size into account)
				x = cr - tw

			self.model.position = (x * self.model.limit) / cw

	def move_thumb (self):
		thumb_rect = self.get_thumb_rect()
		if thumb_rect != self._last_thumb_rect:
			tl,tt,tr,tb = thumb_rect
			cl,ct,cr,cb = self.get_client_rect()
			dc = self.get_dc()
			self.draw_thumb (dc, self.get_thumb_rect())
			brush = windc.get_sys_brush (windc.COLOR_WINDOW)
			dc.fill_rect ((cl,ct,tl-1,tb), brush)
			dc.fill_rect ((tr+1,ct,cr,cb), brush)
			dc.release_dc()
			self._last_thumb_rect = thumb_rect

	def get_thumb_rect (self):
		l,t,r,b = self.get_client_rect()
		w = r-l
		h = b-t

		m = self.model
		pos, lim, pag = m.position, m.limit, m.page

		if lim == 0:
			tp = 0
			tw = w
		else:
			tp = (pos * w) / lim
			tw = (pag * w) / lim

		if tw > w:
			tw = w
		if tw < self.minimum_thumb_size:
			tw = self.minimum_thumb_size
		tx = l+tp
		return tp, t, tx+tw, b

	def draw_thumb (self, dc, rect):
		dc.draw_3d_box (rect, 'etched')
		# add some texture, with a small inset wedge
		l,t,r,b = rect
		offset = 2
		cx = l + ((r-l)/2)
		dc.draw_3d_box (((cx-3), t+offset, (cx+3), b-offset), 'etched')

	def get_preferred_size (self):
		return sys.maxint, self.width

# ===========================================================================
# Scroll Buttons
# ===========================================================================

class scroll_button_model (mvc.auto_model):

	def __init__ (self, awake=0, armed=0, pressed=0, toggled=0, enabled=1, direction='up'):
		mvc.auto_model.__init__ (
			self,
			awake		= awake,
			armed		= armed,
			enabled		= enabled,
			pressed		= pressed,
			toggled		= toggled,
			direction	= direction
			)

	def set_pressed_hook (self, bool):
		if self.enabled:
			return bool
		else:
			return 0

class scroll_button (mvc_button.button):

	def do_paint (self, dc):
		m = self.model
		mvc_button.button.do_paint (self, dc)
		l,t,r,b = self.get_client_rect()
		h = b-t
		w = r-l
		dx = w/3
		dy = h/3

		if m.enabled:
			pen = windc.get_sys_pen (windc.COLOR_WINDOWTEXT)
			brush = windc.get_sys_brush (windc.COLOR_3DSHADOW)
		else:
			pen = windc.get_sys_pen (windc.COLOR_GRAYTEXT)
			brush = windc.get_sys_brush (windc.COLOR_WINDOW)

		dc.select_object (pen)
		dc.select_object (brush)

		md = m.direction
		if md == 'up':
			dc.polygon ((dx, h-dy), (w/2, dy), (w-dx, h-dy))
		elif md == 'down':
			dc.polygon ((dx, dy), (w-dx, dy), (w/2, h-dy))
		elif md == 'left':
			dc.polygon ((dx, h/2), (w-dx, dy), (w-dx, h-dy))
		elif md == 'right':
			dc.polygon ((dx, dy), (dx, h-dy), (w-dx, h/2))
		else:
			raise ValueError, 'Unknown scroll button direction: %s' % repr(md)

	width = 16

	def get_preferred_size (self):
		return self.width, self.width

class vertical_scrollbar_component (layout.vbox):

	model = None
	style = winwin.WS_CHILD | winwin.WS_VISIBLE
	width = 16

	def create (*args):
		self = apply (layout.vbox.create, args)

		if self.model is None:
			self.model = model = scroll_model()
		else:
			model = self.model

		sv = vertical_scrollbar (parent=self).create()
		sv.set_model (model)

		w = self.width

		ubm = scroll_button_model (direction='up')
		dbm = scroll_button_model (direction='down')
		ub = scroll_button(parent=self, width=w).create()
		db = scroll_button(parent=self, width=w).create()
		ub.set_model (ubm)
		db.set_model (dbm)

		# watch range of scrollbar to enable/disable buttons
		model.add_view (self)
		model.add_view (sv)
		ubm.add_view (self)
		dbm.add_view (self)

		self.add (ub)
		self.add (sv)
		self.add (db)

		ub.show_window()
		db.show_window()
		sv.show_window()

		self.buttons = ub, db

		# don't forget about create's responsibility
		return self

	# this is usually appropriate, since our scrollbars can work
	# in units of lines, or whatever.

	button_delta = 1

	def notify (self, model, hint):
		ub, db = self.buttons
		sm, um, dm = self.model, ub.model, db.model

		if model is sm:
			if model.position == 0:
				um.enabled = 0
			else:
				um.enabled = 1

			if model.position + model.page >= model.limit:
				dm.enabled = 0
			else:
				dm.enabled = 1

		elif model is um:
			if um.pressed:
				sm.position = sm.position - self.button_delta

		elif model is dm:
			if dm.pressed:
				sm.position = sm.position + self.button_delta

	def get_preferred_size (self):
		w, h = self.buttons[0].get_preferred_size()
		return w, sys.maxint


class horizontal_scrollbar_component (layout.hbox):

	model = None
	style = winwin.WS_CHILD | winwin.WS_VISIBLE

	width = 16

	# ugly hack for scrollbars meeting in the lower right corner
	add_spacer = 1

	def create (*args):
		self = apply (layout.hbox.create, args)

		if self.model is None:
			self.model = model = scroll_model()
		else:
			model = self.model

		sv = horizontal_scrollbar (parent=self).create()
		sv.set_model (model)

		w = self.width

		lbm = scroll_button_model (direction='left')
		rbm = scroll_button_model (direction='right')
		lb = scroll_button(parent=self, width=w).create()
		rb = scroll_button(parent=self, width=w).create()
		lb.set_model (lbm)
		rb.set_model (rbm)

		# watch range of scrollbar to enable/disable buttons
		model.add_view (self)
		model.add_view (sv)
		lbm.add_view (self)
		rbm.add_view (self)

		self.add (lb)
		self.add (sv)
		self.add (rb)
		if self.add_spacer:
			self.add (self.spacer (self.width, self.width))

		lb.show_window()
		rb.show_window()
		sv.show_window()

		self.buttons = lb, rb

		# don't forget about create's responsibility
		return self

	# this is usually appropriate, since our scrollbars can work
	# in units of lines, or whatever.

	button_delta = 1

	def notify (self, model, hint):
		lb, rb = self.buttons
		sm, lm, rm = self.model, lb.model, rb.model

		if model is sm:
			if model.position == 0:
				lm.enabled = 0
			else:
				lm.enabled = 1

			if model.position + model.page >= model.limit:
				rm.enabled = 0
			else:
				rm.enabled = 1

		elif model is lm:
			if lm.pressed:
				sm.position = sm.position - self.button_delta

		elif model is rm:
			if rm.pressed:
				sm.position = sm.position + self.button_delta

	def get_preferred_size (self):
		w, h = self.buttons[0].get_preferred_size()
		return sys.maxint, h

def test():
	import layout
	c = layout.container('scrollbar test').create()
	c.handle_erase_background = 0
	bl = layout.border_layout (c)
	vs = vertical_scrollbar_component (parent=c).create()
	hs = horizontal_scrollbar_component (parent=c).create()
	vs.model.page = 20
	hs.model.page = 20
	bl.add (vs, 'east')
	bl.add (hs, 'south')
	vs.show_window()
	hs.show_window()
	c.show_window()
	return c,vs,hs

if __name__ == '__main__':
	import msgloop
	test()
	msgloop.go()
