# -*- Mode: Python; tab-width: 4 -*-

# Use a sequence model to capture changes to a treelist.

import treelist
import winclass
import windc
import winfont
import wingdi
import winwin
import winscroll

import mvc
import mvc_seq_win

SPLIT_WORD = winwin.SPLIT_WORD

class tree_window (mvc_seq_win.sequence_window):

	line_height = 16

	black_pen	= windc.get_sys_pen		(windc.COLOR_WINDOWTEXT)
	white_brush	= windc.get_sys_brush	(windc.COLOR_WINDOW)
	limb_pen	= windc.get_sys_pen		(windc.COLOR_ACTIVECAPTION)
	on_brush	= windc.get_sys_brush	(windc.COLOR_ACTIVECAPTION)
	off_brush	= windc.get_sys_brush	(windc.COLOR_INACTIVECAPTION)
	white_color	= windc.get_sys_color	(windc.COLOR_WINDOW)

	def draw_line (self, dc, index):
		[nc, cn, pil, d, node] = self.model[index]

		y = self.line_height * index

		old_pen = dc.select_object (self.limb_pen)

		o = self.line_height
		o2 = o/2
		limbs = self.tree.limbs (index)

		x = o

		# draw vertical connectors
		for limb in limbs:
			if limb:
				dc.move_to ((x+o2, y))
				dc.line_to ((x+o2, y+o))
			x = x + o

		# draw vertical connector to parent
		if limbs and not limbs[-1]:
			dc.move_to ((x-o2, y))
			dc.line_to ((x-o2, y+o2))

		if pil:
			# draw horizontal connector and button
			dc.move_to ((x-o2, y+o2))
			dc.line_to ((x, y+o2))

		if hasattr (node, 'leaf') and not node.leaf:
			br = self.get_button_rect (index)

			if nc:
				# this tree is currently open
				dc.select_object (self.black_pen)
				dc.select_object (self.off_brush)
				self.draw_button (dc, br)
			elif node.has_children:
				# it can be opened, but isn't
				dc.select_object (self.black_pen)
				dc.select_object (self.on_brush)
				self.draw_button (dc, br)
			else:
				# it's an empty non-leaf.
				dc.select_object (self.black_pen)
				dc.select_object (self.white_brush)
				self.draw_button (dc, br)

		node.draw (self, dc, (x, y))
		dc.select_object (old_pen)

	def get_button_rect (self, index):
		[nc, cn, pil, d, node] = self.model[index]
		o = self.line_height
		o2 = o/2
		o4 = o/4
		x = (o * (d+1))
		y = o * index
		return ((x-o2-o4,y+o4,x-o4,y+o2+o4))
		
	def button_hit_test (self, (x, y), index):
		"test a point for a button hit"
		[nc, cn, pil, d, node] = self.model[index]
		o = self.line_height
		bx = (o * d)
		if (bx <= x <= (bx+o)):
			# hit the button
			return 'button'
		elif x > bx:
			# item selection
			return 'selection'
		else:
			return None

	def draw_button (self, dc, button_rectangle):
		dc.round_rect (button_rectangle, 12, 12)

	def WM_LBUTTONDOWN (self, wparam, lparam):
		x, y = self.dp_to_lp (SPLIT_WORD (lparam))
		index = y/self.line_height
		if index < len (self.tree):
			kind = self.button_hit_test ((x, y), index)
			if kind == 'button':
				self.toggle_node (index)
				return 1
			elif kind == 'selection':
				self.select_node (index, self.model[index][4])
		return 0

	def WM_RBUTTONDOWN (self, wparam, lparam):
		x, y = self.dp_to_lp (SPLIT_WORD (lparam))
		index = y/self.line_height
		if index < len (self.tree):
			kind = self.button_hit_test ((x, y), index)
			if kind == 'button':
				self.toggle_node (index, all=1)
				return 1
			elif kind == 'selection':
				self.select_node (index, self.model[index][4], option=1)
		return 0

	selected_node = None

	def select_node (self, index, node, option=0):
		self.invalidate_line (index)
		old_node = self.selected_node
		if old_node:
			old_index = self.tree.get_node_index (old_node)
			self.invalidate_line (old_index)

		if option:
			if hasattr (node, 'option'):
				node.option(self)
		else:
			if hasattr (node, 'select'):
				node.select(self)

		self.selected_node = node
		self.invalidate_line (index)

	def toggle_node (self, index, all=0):
		[nc, cn, pil, d, node] = self.model[index]

		lh = self.line_height
		olt = len (self.tree)

		if nc:
			self.tree.close_node (index)
		else:
			wait = winclass.wait_cursor()
			if all:
				self.tree.open_node_all (index)
			else:
				self.tree.open_node (index)

	def invalidate_line (self, index):
		lh = self.line_height
		y = index * lh
		w, h = self.size
		self.invalidate_rect ((0,y,w, y+lh))

# tree view with scrollbars.

class tree_component (mvc_seq_win.sequence_component):

	sequence_view_class = tree_window
	# it seems more useful on the left side...
	vert_scrollbar_border = 'west'

	add_spacer = 0

	def create (*args):
		self = apply (mvc_seq_win.sequence_component.create, args)
		del self.layout_children[self.horz_scrollbar_border].layout_children[-1]
		return self

def demo():

	class test_path_node (treelist.path_node):

		import os

		line_height = 14
		font_name = 'MS Sans Serif'
		n_font = winfont.font (font_name, height=line_height).create()
		b_font = winfont.font (font_name, height=line_height, weight=winfont.FW_BOLD).create()	

		def draw (self, tree, dc, (x, y)):
			name = self.os.path.split (self.root)[-1]

			if not name:
				name = self.root

			if self.leaf:
				dc.select_object (self.n_font)
			else:
				dc.select_object (self.b_font)

			dc.text_out ((x,y), name)

	sequence = mvc.sequence_model ([test_path_node('c:/')])
	tree = treelist.tree_list (sequence)
	tree.open_node (0)
	w = tree_component (
		'tree test',
		style=winwin.WS_OVERLAPPEDWINDOW,
		w = 300,
		h = 800
		).create()
	w.set_model (sequence)
	w.sequence_window.tree = tree
	w.show_window()
	return w
	
if __name__ == '__main__':
	import msgloop
	w = demo()
	msgloop.go()
	
