# -*- Mode: Python; tab-width: 4 -*-

#
# Simple layout manager support, modeled after the java awt.
#

# Each container window owns a layout manager.  The layout manager
# will position the children according to its policy.
#
# Currently implemented: border_layout and flow_layout.
# TODO: proportional_layout (like that managed by splitter.py)

# TODO: implement an optional property of the border layout that
# either allows different objects in the corners, or knows how to
# decorate the corners.  This would allow us to have MS-style
# scrollbars meet in the bottom right.

# I think that by following the Java AWT model of using a separate
# object to handle layout, we have over-complicated things.  I am
# considering using instead 'layout containers', windows that
# implement the layout policy directly.  This should help with
# chasing cycles and should really simplify the use of the layout
# managers.

import wingdi
import winwin

# returns LOWORD, HIWORD
def SPLIT_WORD (x):
	return (x&0xffff, (x>>16))

class container_mixin:

	def add (self, child, constraint=None):
		self.layout_manager.add (child, constraint)

	size = 100, 100

	def WM_SIZE (self, wparam, lparam):
		self.size = SPLIT_WORD (lparam)
		if lparam:
			self.layout_manager.do_layout()

	def move_child_window (self, child, rect):
		# This is the default action for 'heavyweight' containers.
		apply (child.move_window, rect)

class border_layout:

	def __init__ (self, container):
		self.children = {}
		self.container = container
		# cycle
		self.container.layout_manager = self

	constraints = ('north', 'south', 'east', 'west', 'center')

	def add (self, child, constraint):
		if constraint not in self.constraints:
			raise ValueError, "Unsupported constraint.  Use one of %s" % repr(self.constraints)
		elif self.children.has_key (constraint):
			raise ValueError, "A child has already been positioned there"
		else:
			self.children[constraint] = child

	def get_child (self, wind):
		if self.children.has_key (wind):
			return self.children[wind]
		else:
			return None

	def get_size (self, wind):
		child = self.get_child (wind)
		if child :
			return child.get_preferred_size()
		else:
			return (0, 0)

	def do_layout (self):
		# Lay out each component.  Determine the bounds of the center
		# rectangle, given the preferred sizes of any child windows.
		#
		#   +--------------+
		#   |              |
		#  	+---o------o---+          cl, ct       cr, ct
		#  	|	|  	   |   |
   	   	#	|	|  	   |   |
		#	|	|  	   |   |
		#	+---o------o---+          cl, cb       cr, cb
		#	|			   |
		# 	+--------------+
		#

		cont = self.container
		l,t,r,b = cont.get_client_rect()

		ct = t + self.get_size ('north')[1]
		cb = b - self.get_size ('south')[1]
		cl = l + self.get_size ('west')[0]
		cr = r - self.get_size ('east')[0]

		child = self.get_child ('north')
		if child:
			cont.move_child_window (child, (l,t,r-l,ct-t))

		child = self.get_child ('south')
		if child:
			cont.move_child_window (child, (0,cb,r-l,b-cb))
			
		child = self.get_child ('west')
		if child:
			cont.move_child_window (child, (0,ct,cl-l,cb-ct))
		
		child = self.get_child ('east')
		if child:
			cont.move_child_window (child, (cr,ct,r-cr,cb-ct))

		child = self.get_child ('center')
		if child:
			cont.move_child_window (child, (cl,ct,cr-cl,cb-ct))

class flow_layout:

	def __init__ (self, container):
		self.children = []
		self.container = container
		self.container.layout_manager = self

	def add (self, child, constraint=None):
		# no constraints for now
		self.children.append (child)

	def do_layout (self):
		cont = self.container
		rect = (l,t,r,b) = cont.get_client_rect()
		x, y = l,t

		i = 0

		while i < len(self.children):

			row = []
			tallest = 0

			while i < len(self.children):
				child = self.children[i]
				w, h = child.get_preferred_size()
				# pack until we run out of room on the right
				if ((w + x) > r) and (x != l) :
					break
				else:
					row.append ((child, (w, h)))

				if h > tallest:
					tallest = h

				i = i + 1
				x = x + w

			x = l
			# lay out this row
			for child, (w, h) in row:
				cont.move_child_window (child, (x, y + ((tallest - h) / 2), w, h))
				x = x + w

			y = y + tallest
			x = l


class container (winwin.python_window, container_mixin):

	# If your layout manager will completely cover the container
	# window, then set this to one.  Otherwise, set it to zero.
	handle_erase_background = 1

	def WM_ERASEBKGND (self, wparam, lparam):
		return self.handle_erase_background

	def WM_DESTROY (self, wparam, lparam):
		del self.layout_manager.container
		del self.layout_manager
		return winwin.python_window.WM_DESTROY (self, wparam, lparam)


# We would like to have a container that didn't require a useless
# window from the operating system.

# problems: 
# 1) currently the layout managers directly call move_window() on the
#    child windows, which succeeds because they expect a particular
#    local coordinate system that is inaccurate.
#     
# fix: route move_window calls through the container via a
#    'move_child_window' method.
#     
# 2) when creating 'real' child windows, we need access to the 'real'
#    parent window.  is there a chain?
#     
# fix: 

class lightweight_container:

	def __init__ (self, **d):
		print 'lw.__init__', d

	def move_window (self, x, y, w, h):
		print 'lw_c.move_window() origin=',x,y
		self.origin = x, y
		self.size   = w, h
		self.layout_manager.do_layout()

	def move_child_window (self, child, (x,y,w,h)):
		ox, oy = self.origin
		# adjust coordinates by our origin
		child.move_window (x+ox, y+oy, w, h)

	def get_client_rect (self):
		x, y = self.origin
		w, h = self.size
		print 'gcr',x,y,x+w,y+h
		#return (x, y, x+w, y+h)
		return (0,0,w,h)
		
	def get_real_parent (self):
		p = self.parent
		while not hasattr (p, 'hwnd'):
			p = p.parent
		print 'real_parent() => ', p
		return p
	

def test (kind = 'border'):

	class sample_component (winwin.python_window):

		style = winwin.WS_CHILD | winwin.WS_VISIBLE

		def WM_PAINT (self, wparam, lparam):
			dc = self.begin_paint()
			dc.set_bk_color()
			dc.set_bk_mode()
			rect = l,t,r,b = self.get_client_rect()
			dc.draw_3d_box (rect)
			dc.set_text_align (wingdi.TA_CENTER)
			dc.text_out (
				((r-l)/2,(b-t)/2),
				self.window_name
				)
			self.end_paint()

		def WM_SIZE (self, wparam, lparam):
			self.invalidate_rect (0)

		# this doesn't seem right...
		def WM_MOVE (self, wparam, lparam):
			self.invalidate_rect (0)

		def WM_ERASEBKGND (self, wparam, lparam):
			return 1

		preferred_size = (120, 80)

		def get_preferred_size (self):
			return self.preferred_size

	if kind == 'border':
		parent = frame ('test container').create()
		bl = border_layout (parent)
		for wind in bl.constraints:
			child = sample_component (wind, style=winwin.WS_CHILD)
			child.parent = parent
			parent.add (child, wind)
			child.create()
		parent.show_window()

		for child in bl.children.values():
			child.show_window()

	elif kind == 'flow':
		parent = frame ('test container').create()
		parent.handle_erase_background = 0
		fl = flow_layout (parent)
		info = (
			('one',		(120, 80)),
			('two',		(120, 60)),
			('three',	(200, 80)),
			('four',	(120, 40)),
			('five', 	(200, 50))
			)

		for name, size in info:
			child = sample_component (name, style=winwin.WS_CHILD)
			child.preferred_size = size
			child.parent = parent
			parent.add (child, None)
			child.create()
		parent.show_window()

		for child in fl.children:
			child.show_window()

	else:
		print "Unknown Layout Manager:", kind

if __name__ == '__main__':
	import sys
	import msgloop
	if len(sys.argv) > 1:
		test (sys.argv[1])
	else:
		test ('border')
	msgloop.go()
