# -*- Mode: Python; tab-width: 4 -*-

import layout
import button
import windc
import winwin

SPLIT_WORD = winwin.SPLIT_WORD

class vertical_scrollbar (winwin.python_window):

	scroll_position = 250,1000
	page_size = 250,1000

	style = winwin.WS_CHILD | winwin.WS_VISIBLE

	def WM_SIZE (self, wparam, lparam):
		self.invalidate_rect (0)

	def WM_PAINT (self, wparam, lparam):
		dc = self.begin_paint()
		self.do_paint (dc)
		self.end_paint()

	captured = None

	def WM_LBUTTONDOWN (self, wparam, lparam):
		x, y = SPLIT_WORD (lparam)
		tl, tt, tr, tb = self.get_thumb_rect()
		# did they hit the thumb?
		if (y >= tt) and (y < tb):
			self.captured = y - tt
			self.set_capture()
		# TODO: handle 'page' clicks in the background
		# area.
			
	_last_thumb_rect = None

	def WM_MOUSEMOVE (self, wparam, lparam):
		if self.captured is not None:
			x, y = SPLIT_WORD (lparam)
			# adjust y for offset into thumb
			y = y - self.captured

			tl,tt,tr,tb = self.get_thumb_rect()
			cl,ct,cr,cb = self.get_client_rect()

			# client height
			ch = cb - ct
			# thumb height
			th = tb - tt

			# range check
			if y < 0:
				# top
				y = 0
			elif y + th >= cb:
				# bottom (taking page size into account)
				y = cb - th

			# convert position to n/d
			n, d = self.scroll_position
			self.scroll_position = ((y * d) / ch), d
			self.move_thumb()
			self.scroll_event ()

	def move_thumb (self):
		thumb_rect = self.get_thumb_rect()
		if thumb_rect != self._last_thumb_rect:
			cl,ct,cr,cb = self.get_client_rect()
			tl,tt,tr,tb = thumb_rect
			dc = self.get_dc()
			self.draw_thumb (dc, self.get_thumb_rect())
			brush = windc.get_sys_brush (windc.COLOR_WINDOW)
			dc.fill_rect ((cl,ct,cr,tt-1), brush)
			dc.fill_rect ((cl,tb+1,cr,cb), brush)
			dc.release_dc()
			self._last_thumb_rect = thumb_rect

	def WM_LBUTTONUP (self, wparam, lparam):
		if self.captured is not None:
			self.captured = None
			self.release_capture()

	minimum_thumb_size = 10

	def get_thumb_rect (self):
		l,t,r,b = self.get_client_rect()
		w = r-l
		h = b-t
		# position
		n, d = self.scroll_position
		tp = (n * h) / d
		# page size
		n, d = self.page_size
		th = (n * h) / d
		if th > h:
			th = h
		if th < self.minimum_thumb_size:
			th = self.minimum_thumb_size
		ty = t+tp
		return l, ty, r, ty+th

	callback = None
	def scroll_event (self):
		if self.callback is not None:
			self.callback (self)

	def configure (self, callback, scroll_position=(0,100), page_size=(10,100)):
		self.callback = callback
		self.scroll_position = scroll_position
		self.page_size = page_size

	def draw_thumb (self, dc, rect):
		dc.draw_3d_box (rect)
		# add some texture, with a small inset wedge
		l,t,r,b = rect
		offset = 2
		cy = t + ((b-t)/2)
		dc.draw_3d_box ((l+offset, (cy-3), r-offset, (cy+3)), up=0)

	def do_paint (self, dc):
		back = self.get_client_rect()
		dc.fill_rect (back, windc.get_sys_brush (windc.COLOR_WINDOW))
		self.draw_thumb (dc, self.get_thumb_rect())
		
# most of the code has to be rewritten.  maybe some day we will have a
# fully general coordinate-transform based system, where we need only
# specify a 90-degree turn. 8^(

class horizontal_scrollbar (vertical_scrollbar):

	def WM_LBUTTONDOWN (self, wparam, lparam):
		x, y = SPLIT_WORD (lparam)
		tl, tt, tr, tb = self.get_thumb_rect()
		# did they hit the thumb?
		if (x >= tl) and (x < tr):
			self.captured = x - tl
			self.set_capture()

	def WM_MOUSEMOVE (self, wparam, lparam):
		if self.captured is not None:
			x, y = SPLIT_WORD (lparam)
			# adjust x for offset into thumb
			x = x - self.captured

			tl,tt,tr,tb = self.get_thumb_rect()
			cl,ct,cr,cb = self.get_client_rect()

			# client width
			cw = cr - cl
			# thumb width
			tw = tr - tl

			# range check
			if x < 0:
				# top
				x = 0
			elif x + tw >= cw:
				# bottom (taking page size into account)
				x = cr - tw

			# convert position to n/d
			n, d = self.scroll_position
			self.scroll_position = ((x * d) / cw), d
			self.move_thumb()
			self.scroll_event()

	def move_thumb (self):
		thumb_rect = self.get_thumb_rect()
		if thumb_rect != self._last_thumb_rect:
			tl,tt,tr,tb = thumb_rect
			cl,ct,cr,cb = self.get_client_rect()
			dc = self.get_dc()
			self.draw_thumb (dc, self.get_thumb_rect())
			brush = windc.get_sys_brush (windc.COLOR_WINDOW)
			dc.fill_rect ((cl,ct,tl-1,tb), brush)
			dc.fill_rect ((tr+1,ct,cr,cb), brush)
			dc.release_dc()
			self._last_thumb_rect = thumb_rect

	def get_thumb_rect (self):
		l,t,r,b = self.get_client_rect()
		w = r-l
		h = b-t
		# position
		n, d = self.scroll_position
		tp = (n * w) / d
		# page size
		n, d = self.page_size
		tw = (n * w) / d
		if tw > w:
			tw = w
		if tw < self.minimum_thumb_size:
			tw = self.minimum_thumb_size
		tx = l+tp
		return tp, t, tx+tw, b

	def draw_thumb (self, dc, rect):
		dc.draw_3d_box (rect)
		# add some texture, with a small inset wedge
		l,t,r,b = rect
		offset = 2
		cx = l + ((r-l)/2)
		dc.draw_3d_box (((cx-3), t+offset, (cx+3), b-offset), 0)

class vertical_scrollbar_component (layout.container):

	style = winwin.WS_CHILD | winwin.WS_VISIBLE

	w = 16
	width = w

	def create (*args):
		apply (layout.container.create, args)
		self = args[0]
		self.layout = layout.border_layout (self) # cycle
		thumb = vertical_scrollbar ('vertical scrollbar', parent=self).create()
		ub = button.scroll_button ('scroll up', parent=self).create()
		db = button.scroll_button ('scroll down', parent=self).create()
		ub.preferred_size = self.width, self.width
		db.preferred_size = self.width, self.width
		self.add (ub, 'north')
		self.add (thumb, 'center')
		self.add (db, 'south')
		self.scrollbar = thumb
		self.buttons = ub, db
		return self

	def configure (self, callback, scroll_position=(0,100), page_size=(100,100)):
		self.scrollbar.configure (self.scroll_event, scroll_position, page_size)
		ub, db = self.buttons
		ub.configure ('up', self.scroll_event)
		db.configure ('down', self.scroll_event)
		self.callback = callback

	def set_scroll_position (self, position):
		self.scrollbar.scroll_position = position
		self.scrollbar.invalidate_rect (0)

	def get_scroll_position (self):
		return self.scrollbar.scroll_position

	def set_page_size (self, page_size):
		self.scrollbar.page_size = page_size
		self.scrollbar.invalidate_rect (0)

	def get_page_size (self):
		return self.scrollbar.page_size

	callback = None

	def scroll_event (self, object):
		ub, db = self.buttons
		if object == ub:
			n, d = self.scrollbar.scroll_position
			self.scrollbar.scroll_position = (max (n-1, 0), d)
			self.scrollbar.move_thumb()
			self.last_scroll_position = self.scrollbar.scroll_position
		elif object == db:
			n, d = self.scrollbar.scroll_position
			pn, pd = self.scrollbar.page_size
			# convert page size to n,d units
			page = (pn * d) / pd
			max_n = d - page
			self.scrollbar.scroll_position = (min (n+1, max_n), d)
			self.scrollbar.move_thumb()
			
		if self.callback is not None:
			self.callback (self, self.scrollbar.scroll_position)

	def get_preferred_size (self):
		# find out the size of our parent window
		return self.width, 100

class horizontal_scrollbar_component (vertical_scrollbar_component):

	def create (*args):
		apply (layout.container.create, args)
		self = args[0]
		self.layout = layout.border_layout (self) # cycle
		thumb = horizontal_scrollbar ('horizontal scrollbar', parent=self).create()
		lb = button.scroll_button ('scroll left', parent=self).create()
		rb = button.scroll_button ('scroll right', parent=self).create()
		lb.preferred_size = self.width, self.width
		rb.preferred_size = self.width, self.width
		self.add (lb, 'west')
		self.add (thumb, 'center')
		self.add (rb, 'east')
		self.scrollbar = thumb
		self.buttons = lb, rb
		return self

	def configure (self, callback, scroll_position=(0,100), page_size=(100,100)):
		self.scrollbar.configure (self.scroll_event, scroll_position, page_size)
		lb, rb = self.buttons
		lb.configure ('left', self.scroll_event)
		rb.configure ('right', self.scroll_event)
		self.callback = callback

	def get_preferred_size (self):
		# find out the size of our parent window
		return 100, self.width

def test():

	def callback (bar, scroll_position):
		n, d = scroll_position
		print 'scroll: %08x %3d/%3d' % (id(bar), n, d)

	frame = layout.frame ('frame', style=winwin.WS_OVERLAPPEDWINDOW).create()
	frame.layout = layout.border_layout (frame) # cycle
	frame.handle_erase_background = 0
	vs = vertical_scrollbar_component ('vert', parent=frame, width=16).create()
	vs.configure (callback)

	vs2 = vertical_scrollbar_component ('vert2', parent=frame, width=32).create()
	vs2.configure (callback, (1,5), (2, 5))

	hs = horizontal_scrollbar_component ('horz', parent=frame, width=16).create()
	hs.configure (callback)

	frame.add (vs, 'east')
	frame.add (vs2, 'west')
	frame.add (hs, 'south')

	frame.show_window()

if __name__ == '__main__':
	import msgloop
	test()
	msgloop.go()
