# -*- Mode: Python; tab-width: 4 -*-

# simple text editing window; requires a fixed-width font.

import windc
import winfont
import wingdi
import winwin
import winscroll
import string
import calldll

user32 = windc.user32

SPLIT_WORD = winwin.SPLIT_WORD

class text_edit_window (winscroll.scrollable_window):

	font_height = 16
	font = winfont.font ('Courier New', height=font_height).create()

	def WM_PAINT (self, wparam, lparam):
		dc = self.begin_paint()
		dc.select_object (self.font)
		dc.set_bk_color()
		dc.set_bk_mode (windc.TRANSPARENT)
		dc.set_text_align (wingdi.TA_UPDATECP)

		l,t,r,b = dc.get_clip_box()

		# determine the range of lines we need to draw

		fh = self.font_height
		first = t / fh
		last  = b / fh
		if b % fh:
			last = last + 1

		last = min (len(self.lines), last)

		for i in range (first, last):
			self.draw_line (dc, i*fh, i)

		self.end_paint()

	font_width = 0

	def draw_line (self, dc, y, index):
		dc.move_to ((0, y))
		text = self.lines[index]
		dc.text_out ((0,0), text)
		px, py = dc.move_to_ex ((0,0))

		# this is an ugly but effective way to get the font width
		if not self.font_width:
			if px:
				self.font_width = px/len(text)
				self.show_caret()

		if px > self.widest_line:
			# this isn't right.
			self.widest_line = px
			self.set_virtual_window ((0,0,self.widest_line, self.font_height * len (self.lines)))

	def WM_SIZE (self, wparam, lparam):
		w, h = self.size = SPLIT_WORD (lparam)

		self.width, self.height = w, h
		self.widest_line = 10

		self.set_virtual_window ((
			0,
			0,
			self.widest_line,
			self.font_height * len (self.lines)
			))

		self.set_scroll_values (None, self.font_height)

	current_pos = (0,0)

	def WM_CHAR (self, wparam, lparam):
		# insert / overwrite, etc...
		lh = self.font_height
		ch = chr (wparam)
		w, h = self.size
		x, y = self.current_pos

		line = self.lines[y]

		if ch == '\r':
			if x < len(line):
				self.lines[y] = line[:x]
				self.lines.insert (y+1, line[x:])
			else:
				self.lines.insert (y+1, '')
			self.current_pos = 0, y+1
			self.invalidate_rect ((0,y * lh,w,h))
		else:
			chars = map (None, line)
			chars.insert (x, ch)
			self.lines[y] = string.join (chars, '')
			x = x + 1
			cw = self.font_width
			self.current_pos = x, y
			self.invalidate_rect (
				((x-1) * cw , y * lh, w, (y+1) * lh)
				)

		self.set_caret_pos()

		return 1

	def invalidate_line_range (self, low, high):
		w, h = self.size
		fh = self.font_height
		self.invalidate_rect ((0, low*fh, w, high*fh))

	def WM_DESTROY (self, wparam, lparam):
		winwin.user32.PostQuitMessage (0)

	def WM_SETFOCUS (self, wparam, lparam):
		self.show_caret()
	
	def show_caret (self):
		user32.CreateCaret (self, 0, self.font_width, self.font_height)
		self.set_caret_pos()
		user32.ShowCaret (self)

	def set_caret_pos (self):
		lh = self.font_height
		cw = self.font_width
		x, y = self.current_pos
		x, y = self.lp_to_dp ((x * cw, y * lh))
		user32.SetCaretPos (x, y)

	def WM_KILLFOCUS (self, wparam, lparam):
		user32.DestroyCaret (self)

	def WM_LBUTTONDOWN (self, wparam, lparam):
		mx, my = self.dp_to_lp (SPLIT_WORD (lparam))
		x = mx / self.font_width
		y = my / self.font_height
		y = min (len (self.lines) - 1, y )
		x = min (len (self.lines[y]), x)
		self.current_pos = x, y
		self.set_caret_pos ()

if __name__ == '__main__':

	def frob (line):

		while line and line[-1] in '\r\n':
			line = line[:-1]

		return string.join (string.split (line, '\t'), '    ')

	w = text_edit_window ('text edit demo')
	w.lines = map (frob, open ('textedit.py','rb').readlines())

	w.create()
	w.show_window()

	import msgloop
	msgloop.go()
