# -*- Mode: Python; tab-width: 4 -*-

import npstruct
import string

error = 'npstruct module error'

converter = {
	'b':1,
	'c':1,
	'h':2,
	'l':4
	}	

def calcsize (format):
	size = 0
	i = 1
	while i < len(format):
		mul = 1

		# read a multiplier, if present
		if format[i] in string.digits:
			num_digits = 0
			while format[i] in string.digits:
				num_digits = num_digits+1
				i = i + 1
			mul = string.atoi (format[i-num_digits:i])

		if format[i] == '(':
			end = string.index (format, ')', i)
			bits = reduce (
				lambda x,y: x+ y,
				map (string.atoi, string.split (format[i+1:end]))
				)
			if (bits % 8) != 0:
				raise error, 'bitfield not a multiple of 8 bits %s' % (format)
			size = size + (mul * (bits/8))
			i = end

		elif format[i] == '[':
			# this is a variable-sized struct.  should this ignore the
			# variable parts, or return -1 or something? [how about
			# -s, where s is the size of the fixed part?]  For now, we
			# do not adjust size, and instead require the user to
			# manually adjust the oracle's size.
			i = string.find (format, ']', i+1)

		else:
			size = size + (mul * (converter[format[i]]))

		i = i + 1

	return size

# ---------------------------------------------------------------------------
# an Oracle can be used to divine the contents of mysterious block
# data, using struct-module-like format strings.  an Oracle can also
# write out its data.
# ---------------------------------------------------------------------------
# here's an example from a gif decoder module:
#
# logical_screen_descriptor = npstruct.Oracle (
# 	'Logical Screen Descriptor',
# 	'Lhh(1 3 1 3)bb',
# 	('width',
# 	 'height',
# 	 'global color table flag',
# 	 'color resolution',
# 	 'sort flag',
# 	 'size of global color table',
# 	 'background color index',
# 	 'pixel aspect ratio'))
# 
# >>> logical_screen_descriptor.unpack (strange_data)
# ({'width':100, 'height':100, 'global color table flag':1 ...}, <size_of_struct>)

# 'functions' values are optionally a tuple of two functions,
# one for unpacking, and one for packing.  Otherwise, this expects
# only a reading function.

import types

def get_functions (function_dict):
	rf = {}
	wf = {}
	for key, value in function_dict.items():
		if type(value) == type(()):
			rf[key] = value[0]
			wf[key] = value[1]
		elif type(value) == types.InstanceType and hasattr (value, '_is_oracle'):
			rf[key] = value.procfield_read
			wf[key] = value.procfield_write
		else:
			rf[key] = value
			wf[key] = None
	return rf, wf

class Oracle:

	# crummy type-checking.
	_is_oracle = 1

	def __init__ (self, name, format, names, **functions):
		self.name = name
		self.format = format
		# fixme: need to sanity check len(names) against self.format
		self.names = names
		self.functions = functions
		(self.read_functions, self.write_functions) = get_functions (functions)
		self.size = calcsize (format)

	def __repr__ (self):
		return '<%s oracle>' % self.name

	def new_raw (self):
		return '\000' * self.size

	def new (self):
		return self.unpack (self.new_raw())[0]

	def unpack (self, data, offset=0):
		members, length = npstruct.unpack (self.format, data, offset, self.read_functions)
		result = {}
		names = self.names
		for i in range(len(names)):
			result[names[i]] = members[i]
		return result, length

	def procfield_read (self, results, data, offset):
		r = self.unpack (data, offset)
		# -- tricky singleton tuple --
		return (r[0],), r[1]

	def procfield_write (self, thing):
		return self.pack (thing)

	def pack (self, dict):
		names = self.names
		members = range (len (names))
		for i in members:
			members[i] = dict[names[i]]
		return npstruct.pack (self.format, tuple(members), self.write_functions)

	def describe (self, dict):
		print '%s:' % self.name
		print '--------------------'
		for key in self.names:
			print '%s: %s' % (key, repr(dict[key]))
