# -*- Mode: Python; tab-width: 4 -*-

# Python Interpreter
#
# Features:
#   completion			- (hit the TAB key)
#   input history		- (arrow keys, Ctrl-P, Ctrl-N)
#   clickable objects	- (dangerous, will probably be removed)
#

import calldll

import StringIO
import os
import regex
import string
import sys
import pprint
	
import completion

import mvc
import mvc_cell_win
import region

import windc
import wingdi
import winmesg
import winwin

SPLIT_WORD = winwin.SPLIT_WORD

std_files = None, None, None

class python_interpreter_window (mvc_cell_win.text_edit_window):

	herald_color = (  0,   0, 192)
	input_color =  (  0,   0,   0)
	output_color = (  0, 128,   0)
	error_color =  (192,   0,   0)
	counter = 0

	def create (*args):
		self = apply (mvc_cell_win.text_edit_window.create, args)
		python_interpreter_window.counter = python_interpreter_window.counter + 1
		self.get_parent().set_window_text ('Python Interpreter %d' % self.counter)
		self.set_model (
			mvc.sequence_model (
				self.make_herald()
				)
			)
		self.local_env = {}
		return self

	def prepare_dc (self, dc):
		mvc_cell_win.text_edit_window.prepare_dc (self, dc)
		dc.select_object (self.font)
		dc.set_bk_color()
		dc.set_bk_mode (windc.TRANSPARENT)

	def make_herald (self):
		herald = (
			'Python DynWin Interpreter Copyright 1998 Nightmare Software',
			'Python %s on Win32' % (sys.version),
			sys.copyright,
			)
		result = []
		for line in herald:
			result.append (self.herald_color, line)
		return result

	prompt_text = '>>> '

	def prompt (self, p=None):
		if p is None:
			p = self.prompt_text
		self.model.append ( (self.input_color, p) )
		n = len (self.model)
		self.cursor_pos = (len(p), n-1)
		self.scoot()
		self.set_caret_pos()

	def get_ps2 (self):
		return ('.'*(len(self.prompt_text)-1)) + ' '

	def scoot (self):
		"scroll the window up to make room for output"
		vm, hm = self.scroll_models
		cx, cy = self.get_cell_size()
		w, h = self.size
		n = len (self.model)

		lx, ly = self.get_limits()

		vm.limit = ly
		vm.page = h/cy

		if (h < (n * cy)):
			vm.position = ly - (h/cy)

		if len(self.model):
			hm.limit = lx
			hm.page = w/cx

	def get_limits (self):
		if len(self.model):
			lx, ly = (
				max (map (lambda x: len(x[1]), self.model)),
				len (self.model)
				)
		else:
			lx, ly = 0, 0
		return lx, ly


	def find_repr (self, line, pos):
		# search <line> from <pos> for the common
		# object repr "<kind at a7e983>".  handle
		# embedded reprs, too.

		# search left
		depth = 1
		i = pos
		while (i >= 0) and depth:
			if line[i] == '>':
				depth = depth + 1
			elif line[i] == '<':
				depth = depth - 1
			i = i - 1
			
		if i < 0 and depth:
			return None
		else:
			left = i + 1
			
		# search right
		ll = len(line)
		depth = 1
		i = pos
		while (i < ll) and depth:
			if line[i] == '>':
				depth = depth - 1
			elif line[i] == '<':
				depth = depth + 1
			i = i + 1
		
		if i == ll and depth:
			return None
		else:
			right = i

		return line[left:right], (left, right)
			
	repr_regex = regex.compile ('<.* at \([0-9A-Fa-f]+\)[ ]?>$')

	def conjure_from_repr (self, r):
		reg = self.repr_regex
		if reg.match (r) == len(r):
			try:
				address = string.atoi (reg.group (1), 16)
				return calldll.conjure_object (address)
			except:
				return None

	def WM_LBUTTONDOWN (self, wparam, lparam):
		if self.captured_rect is not None:
			r, o = self.captured_rect
			self.append_lines ([repr(o)], self.output_color)
			self.local_env['_'] = o
			self.prompt()
		else:
			x, y = self.dp_to_lp (SPLIT_WORD (lparam))
			cw, ch = self.get_cell_size()
			cx, cy = x/cw, y/ch
			m = self.model
			lm = len(m)
			if lm and (cy < lm):
				ignore, line = self.model[cy]
				if cx < len(line):
					lp = len (self.prompt_text)
					if line[:lp] == self.prompt_text:
						# we've clicked on an input line.
						# recreate the event.
						self.append_lines ([line], self.input_color)
						self.handle_line_input (line)

	captured_rect = None

	def WM_MOUSEMOVE (self, wparam, lparam):
		x, y = self.dp_to_lp (SPLIT_WORD (lparam))
		cw, ch = self.get_cell_size()
		cx, cy = x/cw, y/ch
		if self.captured_rect is None:
			lm = len(self.model)
			if lm and (cy < lm):
				ignore, line = self.model[cy]
				if (cx < len(line)):
					r = self.find_repr (line, cx)
					if r:
						r, (left, right) = r
						o = self.conjure_from_repr (r)
						if o is not None:
							rect = (left * cw, cy * ch, right * cw, (cy+1)*ch)
							dc = self.get_dc()
							dc.draw_focus_rect (rect)
							dc.release_dc()
							self.captured_rect = rect, o
							self.set_capture()
		else:
			r, o = self.captured_rect
			if not region.point_in_region_p (x, y, r):
				self.release_capture()
				dc = self.get_dc()
				dc.draw_focus_rect (r)
				dc.release_dc()
				self.captured_rect = None

	def WM_KEYDOWN (self, wparam, lparam):
		if wparam == winmesg.VK_UP:
			self.history_delta (+1)
		elif wparam == winmesg.VK_DOWN:
			self.history_delta (-1)
				
	def WM_CHAR (self, wparam, lparam):

		ch = chr (wparam)
		w, h = self.size
		x, y = self.cursor_pos
		cx, cy = self.get_cell_size()

		color, line = self.model[y]

		pl = len(self.prompt_text)

		if ch == '\r':
			# carriage return
			if line:
				self.handle_line_input (line)
				self.line_history_index = 0
			else:
				self.prompt()

		elif ch == '\010':
			# backspace
			self.model[y] = color, line[:-1]
			self.cursor_pos = x-1, y

		elif ch == '\011':
			# tab, completion
			if not self.multi_line:
				self.complete()
			else:
				self.model[y] = ((self.input_color, line + '    '))
				self.cursor_pos = len(line)+4, y
			
		elif ch in ('\020', '\016'):
			# ctrl-p previous line, ctrl-n next line
			if ch == '\020':
				delta = +1
			else:
				delta = -1
			self.history_delta (delta)
		else:
			chars = map (None, line)
			chars.insert (x, ch)
			self.model[y] = ((self.input_color, string.join (chars, '')))
			x = x + 1
			self.cursor_pos = x, y

		self.set_caret_pos()

		return 1

	def history_delta (self, delta):
		if self.line_history:
			x, y = self.cursor_pos
			color, line = self.model[y]
			self.line_history_index = (self.line_history_index + delta) % len(self.line_history)
			line = self.line_history[-self.line_history_index]
			self.model[y] = color, line
			self.cursor_pos = len(line), y
			self.set_caret_pos()

	def draw_line (self, dc, index, range=None):
		cw, ch = self.get_cell_size()
		y = ch * index
		dc.select_object (self.font)
		color, line = self.model[index]
		dc.set_text_color (color)
		if range is None:
			dc.text_out ((0,y), line)
		else:
			l,h = range
			h = h + 1
			dc.text_out ((l*cw, y), line[l:h])


	multi_line = 0

	line_history = None
	line_history_index = 0

	def append_to_history (self, line):
		# manage line history
		if self.line_history is None:
			self.line_history = []

		if self.line_history and self.line_history[-1] == line:
			# don't fill the history with copies of a single line
			pass
		else:
			self.line_history.append (line)

	def handle_line_input (self, line):
		ps2 = self.get_ps2()
		lps2 = len(ps2)
		# check for special case inputs...
		if line == (self.prompt_text):
			# empty input
			self.prompt()
			return
		elif self.multi_line:
			if line == ps2:
				# search for the start:
				ml = []
				lp = len(self.prompt_text)
				m = self.model
				y = len(m) - 2
				while m[y][1][:lps2] == ps2:
					ml.append (m[y][1][lps2:])
					y = y - 1
				ml.append (m[y][1][lp:])
				ml.reverse()
				line = string.join (ml, '\n') + '\n'
			else:
				self.append_to_history (line)
				self.prompt (ps2)
				return				
		elif line[-1] == ':':
			self.multi_line = 1
			self.append_to_history (line)
			self.prompt (ps2)
			return
		else:
			self.append_to_history (line)
			line = line[len(self.prompt_text):]
				
		global std_files
		std_files = sys.stdin, sys.stdout, sys.stderr
		try:
			of = StringIO.StringIO()
			ef = StringIO.StringIO()
			sys.stdout = of
			sys.stderr = ef
			try:
				try:
					if self.multi_line:
						# <ugh>
						raise SyntaxError
					# first try eval
					co = compile (line, repr(self), 'eval')
					result = eval (co, self.local_env)
					if result is not None:
						pprint.pprint (result)
					self.local_env['_'] = result
				except SyntaxError:
					# now try exec
					co = compile (line, repr(self), 'exec')
					exec co in self.local_env
			except:
				self.write_traceback (
					sys.exc_type,
					sys.exc_value,
					sys.exc_traceback,
					sys.stderr
					)
		finally:
			sys.stdin, sys.stdout, sys.stderr = std_files

		self.multi_line = 0

		ef.seek (0)
		lines = ef.readlines()
		self.append_lines (lines, self.error_color)
		of.seek (0)
		lines = of.readlines()
		self.append_lines (lines, self.output_color)
		self.prompt()


	complete_chars = string.letters + string.digits + '_.'

	def complete (self):
		# try to complete a partially-typed symbol...
		color, line = self.model[-1]

		i = len(line) - 1
		while i and line[i] in self.complete_chars:
			i = i - 1

		part = line[i+1:]

		if not part:
			winwin.user32.MessageBeep (0)
		else:
			# search local symbols
			prefix, matches = completion.completions (part, self.local_env)
			if len(matches) == 1:
				sym = matches[0]
				line = line[:i+1] + sym
				self.model[-1] = (color, line)
				x, y = self.cursor_pos
				self.cursor_pos = len(line), y
				self.set_caret_pos()
			else:
				line = line + prefix
				lines = ['completions: ' + string.join (matches, ', '), line]
				self.append_lines (lines, self.input_color)
				x, y = self.cursor_pos
				self.cursor_pos = len(line), y+2
				self.set_caret_pos()
				winwin.user32.MessageBeep (0)

	def append_lines (self, lines, color):
		n = len (self.model)
		r = []

		for line in lines:
			r.append (color, self.frob_line (line))

		self.model[n:n] = r
		self.scoot()
		
	def frob_line (self, line):

		while line and line[-1] in '\r\n':
			line = line[:-1]

		return string.join (string.split (line, '\t'), '    ')

	def flush_output (self):
		self.model[:] = []
		self.scoot()

	def write_traceback (self, t, v, tb, file):
		tbinfo = []
		file.write ('Traceback (innermost last)\n')
		file.write ('%s: %s\n' % (t,v))
		r = repr(self)
		while 1:
			filename = tb.tb_frame.f_code.co_filename
			if filename == r:
				filename = "<stdin>"
			file.write (
				'  File "%s", line %d in %s\n' % (
					filename,
					tb.tb_lineno,
					tb.tb_frame.f_code.co_name
					)
				)
			tb = tb.tb_next
			if not tb:
				break

class python_interpreter_component (mvc_cell_win.text_edit_component):

	sequence_view_class = python_interpreter_window

	def prompt (self):
		self.sequence_window.prompt()

	def flush_output (self):
		self.sequence_window.flush_output()

def new_window ():
	w = python_interpreter_component(
		style = winwin.WS_OVERLAPPEDWINDOW
		)
	w.create()
	w.show_window()
	w.prompt()
	return w

if __name__ == '__main__':
	new_window()
	import msgloop
	msgloop.go()
