# -*- Mode: Python; tab-width: 4 -*-

# splitter windows
# 
# <n> windows, split horizontally or vertically, and embedded
# in the parent window.  This window manages the splitter 'bars',
# resizing, etc.
#
# We'd like to be able to add and subtract windows dynamically.

# This is a little clumsy, I have some ideas for cleaning it up, for
# example taking advantage of the fact that we will only get mouse
# events over the wedges.
#
# I think there's a one-off problem in placing the windows, they tend
# to obscure the wedges.

# TODO: implement eagerly-armed behavior, make etched wedges, only
# raise them when the mouse is over them.

import winclass
import windc
import wingdi
import winwin

# returns LOWORD, HIWORD

def SPLIT_WORD (x):
	return (x&0xffff, (x>>16))

# utility functions for manipulating split data

def scale_list (l, num, den):
	"scale a list of numbers by the ratio: num/den"
	r = []
	for n in l:
		r.append ((n * num)/den)
	return r

def get_widths (l, w):
	"convert a list of wedge positions to window widths"
	x = 0
	r = []
	for p in l:
		r.append (p-x)
		x = p
	r.append (w-x)
	return r

def get_wedges (l):
	"convert a list of window widths to wedge positions"
	x = 0
	r = []
	for i in range(1,len(l)):
		x = x + l[i-1]
		r.append (x)
	return r

def insert_window_into_wedges (wl, w, cw, pos):
	"insert child window of width <cw> in wedge-list <wl> at position <pos>"
	# rescale the wedge sizes
	wl = scale_list (wl, w-cw, w)
	# convert to widths
	wl = get_widths (wl, w-cw)
	# insert the new child
	wl.insert (pos, cw)
	# convert back to wedges
	return get_wedges (wl)

class h_splitter (winwin.python_window):

	children = []
	wedges = []
	wedge_width = 6

	split_dim = 0

	def insert (self, child_window, pos=0, percent=50):
		"insert a child window into this splitter, at position <pos>"

		if not self.children:
			self.children = []
			self.wedges = []

		self.children.insert (pos, child_window)
		child_window.set_parent (self)

		n = len (self.children)

		ss = self.size[self.split_dim]

		# how many pixels will we take?
		part = (ss * percent) / 100

		if n > 1:
			self.wedges = insert_window_into_wedges (
				self.wedges, ss, part, pos
				)

		self.place_windows()
		self.invalidate_rect (0)

	def recalc_layout (self, old_size, new_size):
		ow, oh = old_size
		nw, nh = new_size
		# scale the layout by the new width
		self.wedges = scale_list (
			self.wedges,
			new_size[self.split_dim],
			old_size[self.split_dim]
			)
		self.place_windows()
		self.invalidate_rect (0)

	def place_windows (self):
		w, h = self.size
		lp = 0
		ww = self.wedge_width
		# move the windows into place
		for i in range (len (self.wedges)):
			child = self.children[i]
			wx = self.wedges[i]
			width = wx - lp
			child.move_window (lp,0,width,h,1)
			lp = wx + ww
		# move the last window into place
		self.children[-1].move_window (lp,0,w-lp,h,1)

	def WM_PAINT (self, wparam, lparam):
		dc = self.begin_paint()
		self.draw_wedges (dc)
		self.end_paint()

	def draw_wedges (self, dc):
		"override me!"
		(w, h) = self.size
		ww = self.wedge_width
		for wx in self.wedges:
			dc.draw_3d_box ((wx,0,wx+ww-1,h-1))

	temp_cursor = None
	captured_point = None
	moving = 0

	def WM_MOUSEMOVE (self, wparam, lparam):
		(x, y) =  SPLIT_WORD (lparam)
		(w, h) = self.size
		ww = self.wedge_width

		wedges = self.wedges
		hit = 0

		if self.moving:
			n = self.captured_wedge
			if x != wedges[n]:
				dx = x - wedges[n]
				wedges[n] = x

				# move/resize the windows on either side of the wedge
				# find new pos, width of each window

				if n == 0:
					# leftmost wedge
					lp = 0
					lw = x
				else:
					lp = wedges[n-1] + ww
					lw = (x - wedges[n-1]) - ww

				if n == (len(wedges) - 1):
					# rightmost wedge
					rp = x + ww
					rw = (w - x) - ww
				else:
					rp = x + ww
					rw = (wedges[n+1] - x) - ww
					
				self.children[n+1].move_window (rp,0,rw,h,1)
				self.children[n].move_window (lp,0,lw,h,1)
				# redraw wedges...
				self.update_window()
		else:
			n = 0
			for wx in wedges:
				if wx <= x <= (wx+ww):
					if self.temp_cursor is None:
						self.temp_cursor = winclass.temp_cursor (winclass.IDC_SIZEWE)
						self.set_capture()
					self.captured_wedge = n
					hit = 1
					break
				n = n + 1

			if (not hit) and self.temp_cursor:
				self.release_capture()
				self.temp_cursor.off()
				self.temp_cursor = None
				self.captured_point = None

	def WM_LBUTTONDOWN (self, wparam, lparam):
		if self.temp_cursor:
			self.moving = 1

	def WM_LBUTTONUP (self, wparam, lparam):
		if self.moving:
			self.moving = 0

	size = (100,100)
	def WM_SIZE (self, wparam, lparam):
		(w, h) = SPLIT_WORD (lparam)
		if w:
			old_size, self.size = self.size, (w, h)
			if self.children:
				self.recalc_layout (old_size, (w, h))
				self.invalidate_rect (0)


class v_splitter (h_splitter):

	split_dim = 1

	def place_windows (self):
		w, h = self.size
		lp = 0
		ww = self.wedge_width
		# move the windows into place
		for i in range (len (self.wedges)):
			child = self.children[i]
			wy = self.wedges[i]
			height = wy - lp
			child.move_window (0,lp,w,height)
			lp = wy + ww
		# move the last window into place
		self.children[-1].move_window (0,lp,w,h-lp)

	def draw_wedges (self, dc):
		"override me!"
		(w, h) = self.size
		ww = self.wedge_width
		for wy in self.wedges:
			dc.draw_3d_box ((0,wy,w-1,wy+ww-1))

	def WM_MOUSEMOVE (self, wparam, lparam):
		(x, y) =  SPLIT_WORD (lparam)
		(w, h) = self.size
		ww = self.wedge_width

		wedges = self.wedges
		hit = 0

		if self.moving:
			n = self.captured_wedge
			if y != wedges[n]:
				dy = y - wedges[n]
				wedges[n] = y

				# move/resize the windows on either side of the wedge
				# find new pos, width of each window

				if n == 0:
					# top wedge
					tp = 0
					th = y
				else:
					tp = wedges[n-1] + ww
					th = (y - wedges[n-1]) - ww

				if n == (len(wedges) - 1):
					# bottom wedge
					bp = y + ww
					bh = (h - y) - ww
				else:
					bp = y + ww
					bh = (wedges[n+1] - y) - ww
					
				self.children[n+1].move_window (0,bp,w,bh,1)
				self.children[n].move_window (0,tp,w,th,1)

				if dy > 0:
					self.invalidate_rect ((0,y-dy,w,y+ww+dy))
				else:
					self.invalidate_rect ((0,y,w,(y+ww)-dy))
				self.update_window()
		else:
			n = 0
			for wy in wedges:
				if wy <= y <= (wy+ww):
					if self.temp_cursor is None:
						self.temp_cursor = winclass.temp_cursor (winclass.IDC_SIZENS)
						self.set_capture()
					self.captured_wedge = n
					hit = 1
					break
				n = n + 1

			if (not hit) and self.temp_cursor:
				self.release_capture()
				self.temp_cursor.off()
				self.temp_cursor = None
				self.captured_point = None


if __name__ == '__main__':


	class test_window (winwin.python_window):

		def create (self, parent):
			self.style  = winwin.WS_VISIBLE | winwin.WS_CHILD
			self.parent = parent
			return winwin.python_window.create (self)

		def WM_SIZE (self, wparam, lparam):
			(w, h) = SPLIT_WORD (lparam)
			self.size = (w, h)
			self.invalidate_rect (0)

		def WM_PAINT (self, wparam, lparam):
			dc = self.begin_paint()
			w, h = self.size
			dc.set_bk_mode()
			dc.text_out ((10,10), self.window_name)
			dc.draw_3d_box ((w/2-20,h/2-40,w/2+20,h/2),1)
			dc.draw_3d_box ((w/2-20,h/2+20,w/2+20,h/2+60),0)
			self.end_paint()

	import sys

	if len(sys.argv) > 1:
		if sys.argv[1][0] == 'h':
			hs = h_splitter ('horizontal splitter').create()
			w1 = test_window ('test 1').create(hs)
			w2 = test_window ('test 2').create(hs)
			hs.show_window()
			hs.insert (w1)
			hs.insert (w2, 0, 70)
		else:
			vs = v_splitter ('vertical splitter').create()
			w1 = test_window ('test 1').create(vs)
			w2 = test_window ('test 2').create(vs)
			vs.show_window()
			vs.insert (w1)
			vs.insert (w2, 0, 80)

	else:
		hs = h_splitter ('horizontal splitter').create()
		vs = v_splitter ('vertical splitter')
		vs.style = winwin.WS_VISIBLE | winwin.WS_CHILD
		vs.parent = hs
		vs.create()

		w1 = test_window ('test 1').create(vs)
		w2 = test_window ('test 2').create(vs)
		w3 = test_window ('test 3').create(vs)

		w4 = test_window ('test 4').create(hs)
		hs.show_window()

		hs.insert (w4)
		hs.insert (vs)

		vs.show_window()
		vs.insert (w1)
		vs.insert (w2)
		vs.insert (w3)

	import msgloop
	msgloop.go()
