# -*- Mode: Python; tab-width: 4 -*-
# 	$Id: treelist.py,v 1.10 1996/03/15 05:05:34 rushing Exp rushing $	
# GUI-independent

# represent a tree structure as a 'flattened' list.
# a node's children follow it immediately in the list.
# opening a node will insert that node's children after
# it, and closing a node will remove them.

import string

# this is probably slower than "l = map (fun, l)",
# but is necessary in order for hookseq to work.
# [hookseq.hooked_sequence is a sequence container that
#  can 'hook' modifications to the sequence]

def in_place_map (fun, seq):
	for i in range(len(seq)):
		seq[i] = apply (fun, (seq[i],))

tree_list_error = "tree list error"

# FIXME: make this act like a list, rather than containing it?
# Q: can the same node belong to several trees?

# each element is a list:
# [nc cn pil d node]
# nc	== number of children
# cn	== child number <cn> of its parent
# pil	== parent information list
# d		== depth
# node	== the node object
#
# this list is permanently associated with the node.
# so everything is careful to modify the list in-place.

class tree_list:
	def __init__ (self, list=None, keep_node_dict=0):
		# optionally allows us to quickly associate a node object
		# with its <info_list>
		self.keep_node_dict = keep_node_dict
		if keep_node_dict:
			self.node_dict = {}
		if list == None:
			list = []
		else:
			self.list = list
		self.new_roots (list)		

	# ==================================================
	# sequence overrides
	# ==================================================

	def __getitem__ (self, index):
		return self.list[index]

	def __len__ (self):
		return len (self.list)

	# ==================================================

	def new_roots (self, list):
		new_list = list[:]
		for y in xrange (len (new_list)):
			new_list[y] = info_list = [0, y, None, 0, new_list[y]]
			if self.keep_node_dict:
				self.node_dict[info_list[4]] = info_list
		self.list[:] = new_list

	def open_node (self, index):
		info = self.list[index]
		(nc, cn, pil, d, node) = tuple(info)
		children = node.get_children()
		cl = len (children)
		# open only if not already open
		# and there are children
		if not nc and cl:
			# update the number of children
			info[0] = cl
			# insert the children
			for j in range(cl):
				children[j] = info_list = [0, j, info, d+1, children[j]]
				if self.keep_node_dict:
					# add the new children to the node->info_list map
					self.node_dict[info_list[4]] = info_list
			self.list[index:index+1] = [info] + children
			return 1
		else:
			return 0

	# recursively generate the children of this node,
	# and return a list ready for insertion in its place.

	def get_all_children (self, info):
		(nc, cn, pil, d, node) = tuple(info)
		children = node.get_children()
		if not children:
			return [info]
		else:
			result = [info]
			result[0][0] = len(children)
			for y in range (len (children)):
				child_info = [0, y, info, d+1, children[y]]
				list = self.get_all_children (child_info)
				result = result + list
		return result

	# open this node (recursively)
	def open_node_all (self, index):
		children = self.get_all_children (self.list[index])
		if self.keep_node_dict:
			# add the new children to the node->info_list map
			for x in children:
				self.node_dict[x[4]] = x
		self.list[index:index+1] = children

	def close_node (self, index):
		info = self.list[index]
		my_depth = info[3]
		end = index + 1
		while end < len(self.list):
			if self.list[end][3] <= my_depth:
				break
			end = end + 1
		info[0] = 0
		if self.keep_node_dict:
			# remove from the node->info_list map
			for x in self.list[index+1:end]:
				del self.node_dict[x[4]]
		self.list[index:end] = [info]

	# find the index of a particular node
	def get_node_index (self, node):
		if self.keep_node_dict:
			if self.node_dict.has_key (node):
				# NOTE: this is an O(n) operation!
				return self.list.index(self.node_dict[node])
			else:
				return -1
		else:
			for j in range (len (self.list)):
				if self.list[j][4] == node:
					return j
			return -1

	# calculate the limb structure for this index
	def limbs (self, index):
		return (limbs (self.list[index]))

	# print an ascii rep of the tree.
	def print_tree (self):
		for y in range(len(self.list)):
			(nc, cn, pil, d, node) = tuple(self.list[y])			
			l = string.joinfields (map (ascii_limb, self.limbs (y)), '')
			print '%03d %s%s' % (y, l, node)

	# ==================================================
	# these functions are used to asynchronously add
	# items to the tree.
	# ==================================================

	def add_children_index (self, index, children):
		c = children[:]
		info = self.list[index]
		(nc, cn, pil, d, node) = tuple(info)
		cl = len(children)
		knd = self.keep_node_dict
		if knd:
			nd = self.node_dict
		for j in range(cl):
			c[j] = [0, j+nc, info, d+1, c[j]]
			if knd:
				# update the node->info_list map
				nd[c[4]] = c
		self.list[index+1+nc:index+1+nc] = c
		info[0] = cl + nc

	def add_child_index (self, index, child):
		info = self.list[index]
		(nc, cn, pil, d, node) = tuple(info)
		c = [0, nc+1, info, d+1, child]
		if self.keep_node_dict:
			# update the node->info_list map
			self.node_dict[c[4]] = c
		self.list[index+nc+1:index+nc+1] = [c]
		info[0] = nc + 1

	def add_children (self, node, children):
		index = self.get_node_index (node)
		return add_children_index (index, children)

	def add_child (self, node, child):
		index = self.get_node_index (node)
		return add_child (index, child)

# drawing the limbs: traveling back up the tree, whenever
# a parent is _not_ the final child, draw a line segment.
# we need to pass this information on to the draw_line method

def limbs (info):
	r = []
	while info[2] != None:
		if ((info[1]+1) < info[2][0]):
			r.insert (0,1)
		else:
			r.insert (0,0)
		info = info[2]
	return r

def ascii_limb (yesno):
	return (yesno and ('| ')) or '  '

# example node implementation

import os
class path_node:

	def __init__ (self, root, leaf=0):
		self.root = root
		# am I a leaf node?
		if os.path.isdir (root):
			try:
				self.has_children = len(os.listdir(root)) > 0
			except:
				self.has_children = 0
			self.leaf = 0
		else:
			self.has_children = 0
			self.leaf = 1

	# returns a list of nodes
	def get_children (self):
		if not self.leaf:
			try:
				l = os.listdir (self.root)
			except:
				return []

			# make them full pathnames
			l.sort()
			in_place_map (lambda x,s=self: os.path.join (s.root, x), l)
			
			# now make them nodes..
			in_place_map (lambda x,s=self: s.__class__ (x,), l)
			return l
		else:
			return []

	def __repr__ (self):
		return '<path node "%s" children?:%d leaf:%d>' % \
			   (self.root, self.has_children, self.leaf)

# convenient test node class - each child expands into
# four children, and goes four deep.
# To use, try: t = tree_list ([fours_node([0])])

import os
class fours_node:

	def __init__ (self, root, leaf=0):
		self.root = root
		if len (self.root) > 3:
			self.leaf = 1
		else:
			self.leaf = 0

		self.has_children = not self.leaf

	def get_children (self):
		if not self.leaf:
			l = map (lambda x, s=self: s.root + [x], range(4))
			return map (lambda x,c=self.__class__: c(x,), l)
		else:
			return []

	def __repr__ (self):
		return string.joinfields (map(str,self.root), '.')

def test (path):
	tree = tree_list ( [ path_node (path) ] )
	return tree

if __name__ == '__main__':
	import sys
	tree = test (sys.argv[1])
	tree.open_node_all (0)
	tree.print_tree()
