# -*- Mode: Python; tab-width: 4 -*-

# Simple 'Tree Control'

import treelist
import windc
import winfont
import wingdi
import winwin
import winscroll

SPLIT_WORD = winwin.SPLIT_WORD

def get_sys_pen (color):
	r,g,b = windc.unpack_color (windc.user32.GetSysColor (color))
	return wingdi.pen (red=r,green=g, blue=b).create()

def get_sys_brush (color):
	r,g,b = windc.unpack_color (windc.user32.GetSysColor (color))
	return wingdi.brush (red=r,green=g, blue=b).create()

class tree_window (winscroll.scrollable_window):

	line_height = 16

	black_pen	= get_sys_pen 	(windc.COLOR_WINDOWTEXT)
	white_brush	= get_sys_brush	(windc.COLOR_WINDOW)
	limb_pen	= get_sys_pen	(windc.COLOR_ACTIVECAPTION)
	on_brush	= get_sys_brush	(windc.COLOR_ACTIVECAPTION)
	off_brush	= get_sys_brush	(windc.COLOR_INACTIVECAPTION)

	def create (self):
		result = winscroll.scrollable_window.create (self)
		self.set_virtual_window ((0, 0, self.width, self.line_height * len (self.tree)))
		return result

	def WM_PAINT (self, wparam, lparam):
		try:
			dc = self.begin_paint()

			dc.set_bk_color ()
			dc.set_bk_mode (windc.TRANSPARENT)

			l,t,r,b = dc.get_clip_box()
			dc.fill_rect ((l,t,r,b), self.white_brush)
			# determine the range of lines we need to draw

			lh = self.line_height
			first = t / lh
			last  = b / lh
			if b % lh:
				last = last + 1

			last = min (len (self.tree), last)

			for i in range (first, last):
				self.draw_line (dc, i*lh, i)

		finally:
			self.end_paint()

	def draw_line (self, dc, y, index):
		[nc, cn, pil, d, node] = self.tree[index]

		old_pen = dc.select_object (self.limb_pen)

		o = self.line_height
		o2 = o/2
		limbs = self.tree.limbs (index)

		x = o

		# draw vertical connectors
		for limb in limbs:
			if limb:
				dc.move_to ((x+o2, y))
				dc.line_to ((x+o2, y+o))
			x = x + o

		# draw vertical connector to parent
		if limbs and not limbs[-1]:
			dc.move_to ((x-o2, y))
			dc.line_to ((x-o2, y+o2))

		if pil:
			# draw horizontal connector and button
			dc.move_to ((x-o2, y+o2))
			dc.line_to ((x, y+o2))

		if hasattr (node, 'leaf') and not node.leaf:
			br = self.get_button_rect (index)

			if nc:
				# this tree is currently open
				dc.select_object (self.black_pen)
				dc.select_object (self.off_brush)
				self.draw_button (dc, br)
			elif node.has_children:
				# it can be opened, but isn't
				dc.select_object (self.black_pen)
				dc.select_object (self.on_brush)
				self.draw_button (dc, br)
			else:
				# it's an empty non-leaf.
				dc.select_object (self.black_pen)
				dc.select_object (self.white_brush)
				self.draw_button (dc, br)

		node.draw (self, dc, (x, y))
		dc.select_object (old_pen)

	def get_button_rect (self, index):
		[nc, cn, pil, d, node] = self.tree[index]
		o = self.line_height
		o2 = o/2
		o4 = o/4
		x = (o * (d+1))
		y = o * index
		return ((x-o2-o4,y+o4,x-o4,y+o2+o4))
		
	def button_hit_test (self, (x, y), index):
		"test a point for a button hit"
		[nc, cn, pil, d, node] = self.tree[index]
		o = self.line_height
		bx = (o * d)
		if (bx <= x <= (bx+o)):
			# hit the button
			return 'button'
		elif x > bx:
			# item selection
			return 'selection'
		else:
			return None

	def draw_button (self, dc, button_rectangle):
		dc.round_rect (button_rectangle, 12, 12)

	def WM_LBUTTONDOWN (self, wparam, lparam):
		x, y = self.dp_to_lp (SPLIT_WORD (lparam))
		index = y/self.line_height
		if index < len (self.tree):
			kind = self.button_hit_test ((x, y), index)
			if kind == 'button':
				self.toggle_node (index)
				return 1
			elif kind == 'selection':
				self.select_node (index, self.tree[index][4])
		return 0

	selected_node = None

	def select_node (self, index, node):
		self.invalidate_line (index)
		old_node = self.selected_node
		if old_node:
			old_index = self.tree.get_node_index (old_node)
			self.invalidate_line (old_index)
		self.selected_node = node
		self.invalidate_line (index)

	def toggle_node (self, index):
		[nc, cn, pil, d, node] = self.tree[index]

		lh = self.line_height
		olt = len (self.tree)

		if nc:
			self.tree.close_node (index)
		else:
			self.tree.open_node (index)

		nlt = len (self.tree)
		
		if nlt != olt:
			self.set_virtual_window ((0, 0, self.width, lh * nlt))
			self.invalidate_rect ((0, lh * index, self.width, lh * max (olt, nlt)))

	def invalidate_line (self, index):
		lh = self.line_height
		y = index * lh
		self.invalidate_rect ((0,y,self.width, y+lh))

	def WM_SIZE (self, wparam, lparam):
		winscroll.scrollable_window.WM_SIZE (self, wparam, lparam)
		self.invalidate_rect (0)

if __name__ == '__main__':
	import os
	class test_path_node (treelist.path_node):

		line_height = 14
		font_name = 'MS Sans Serif'
		n_font = winfont.font (font_name, height=line_height).create()
		b_font = winfont.font (font_name, height=line_height, weight=winfont.FW_BOLD).create()	

		def draw (self, tree, dc, (x, y)):
			name = os.path.split (self.root)[-1]

			if not name:
				name = self.root

			if self.leaf:
				dc.select_object (self.n_font)
			else:
				dc.select_object (self.b_font)

			dc.text_out ((x,y), name)

	tree = treelist.tree_list ([test_path_node('c:/')])
	tree.open_node (0)
	w = tree_window ('tree test')
	w.tree = tree
	w.create()
	w.show_window()
	import msgloop
	msgloop.go()
	
