# -*- Mode: Python; tab-width: 4 -*-

# Win32-level file operations, including Asynchronous I/O,
# memory mapping, etc...

import windll
import winerror
import structob
import gencb

kernel32 = windll.module ('kernel32')
cstring = windll.cstring

# ===========================================================================
# from <winnt.h>

########################################################################
##                                                                    ##
##                             ACCESS MASK                            ##
##                                                                    ##
########################################################################

##
##  Define the access mask as a longword sized structure divided up as
##  follows:
##
##       3 3 2 2 2 2 2 2 2 2 2 2 1 1 1 1 1 1 1 1 1 1
##       1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0
##      +---------------+---------------+-------------------------------+
##      |G|G|G|G|Res'd|A| StandardRights|         SpecificRights        |
##      |R|W|E|A|     |S|               |                               |
##      +-+-------------+---------------+-------------------------------+
##
##      typedef struct _ACCESS_MASK {
##          WORD   SpecificRights;
##          BYTE  StandardRights;
##          BYTE  AccessSystemAcl : 1;
##          BYTE  Reserved : 3;
##          BYTE  GenericAll : 1;
##          BYTE  GenericExecute : 1;
##          BYTE  GenericWrite : 1;
##          BYTE  GenericRead : 1;
##      } ACCESS_MASK;
##      typedef ACCESS_MASK *PACCESS_MASK;
##
##  but to make life simple for programmer's we'll allow them to specify
##  a desired access mask by simply OR'ing together mulitple single rights
##  and treat an access mask as a DWORD.  For example
##
##      DesiredAccess = DELETE | READ_CONTROL
##
##  So we'll declare ACCESS_MASK as DWORD
##

########################################################################
##                                                                    ##
##                             ACCESS TYPES                           ##
##                                                                    ##
########################################################################


## begin_ntddk begin_nthal begin_ntifs
##
##  The following are masks for the predefined standard access types
##

DELETE                           = (0x00010000L)
READ_CONTROL                     = (0x00020000L)
WRITE_DAC                        = (0x00040000L)
WRITE_OWNER                      = (0x00080000L)
SYNCHRONIZE                      = (0x00100000L)

STANDARD_RIGHTS_REQUIRED         = (0x000F0000L)

STANDARD_RIGHTS_READ             = (READ_CONTROL)
STANDARD_RIGHTS_WRITE            = (READ_CONTROL)
STANDARD_RIGHTS_EXECUTE          = (READ_CONTROL)

STANDARD_RIGHTS_ALL              = (0x001F0000L)

SPECIFIC_RIGHTS_ALL              = (0x0000FFFFL)

##
## AccessSystemAcl access type
##

ACCESS_SYSTEM_SECURITY           = (0x01000000L)

##
## MaximumAllowed access type
##

MAXIMUM_ALLOWED                  = (0x02000000L)

FILE_READ_DATA				= ( 0x0001 )	# file & pipe
FILE_LIST_DIRECTORY			= ( 0x0001 )	# directory

FILE_WRITE_DATA				= ( 0x0002 )	# file & pipe
FILE_ADD_FILE				= ( 0x0002 )	# directory

FILE_APPEND_DATA			= ( 0x0004 )	# file
FILE_ADD_SUBDIRECTORY		= ( 0x0004 )	# directory
FILE_CREATE_PIPE_INSTANCE	= ( 0x0004 )	# named pipe

FILE_READ_EA				= ( 0x0008 )	# file & directory

FILE_WRITE_EA				= ( 0x0010 )	# file & directory

FILE_EXECUTE				= ( 0x0020 )	# file
FILE_TRAVERSE				= ( 0x0020 )	# directory

FILE_DELETE_CHILD			= ( 0x0040 )	# directory

FILE_READ_ATTRIBUTES		= ( 0x0080 )	# all

FILE_WRITE_ATTRIBUTES		= ( 0x0100 )	# all

FILE_ALL_ACCESS 			= (STANDARD_RIGHTS_REQUIRED | SYNCHRONIZE | 0x1FF)

FILE_GENERIC_READ	= (
	STANDARD_RIGHTS_READ
	|FILE_READ_DATA
	|FILE_READ_ATTRIBUTES
	|FILE_READ_EA
	|SYNCHRONIZE
	)

FILE_GENERIC_WRITE	= (
	STANDARD_RIGHTS_WRITE
	| FILE_WRITE_DATA
	| FILE_WRITE_ATTRIBUTES
	| FILE_WRITE_EA
	| FILE_APPEND_DATA
	| SYNCHRONIZE
	)

FILE_GENERIC_EXECUTE = (
	STANDARD_RIGHTS_EXECUTE
	| FILE_READ_ATTRIBUTES
	| FILE_EXECUTE
	| SYNCHRONIZE
	)

FILE_SHARE_READ					= 0x00000001
FILE_SHARE_WRITE				= 0x00000002
FILE_SHARE_DELETE				= 0x00000004
FILE_ATTRIBUTE_READONLY			= 0x00000001
FILE_ATTRIBUTE_HIDDEN			= 0x00000002
FILE_ATTRIBUTE_SYSTEM			= 0x00000004
FILE_ATTRIBUTE_DIRECTORY		= 0x00000010
FILE_ATTRIBUTE_ARCHIVE			= 0x00000020
FILE_ATTRIBUTE_NORMAL			= 0x00000080
FILE_ATTRIBUTE_TEMPORARY		= 0x00000100
FILE_ATTRIBUTE_COMPRESSED		= 0x00000800
FILE_ATTRIBUTE_OFFLINE			= 0x00001000
FILE_NOTIFY_CHANGE_FILE_NAME	= 0x00000001
FILE_NOTIFY_CHANGE_DIR_NAME		= 0x00000002
FILE_NOTIFY_CHANGE_ATTRIBUTES	= 0x00000004
FILE_NOTIFY_CHANGE_SIZE			= 0x00000008
FILE_NOTIFY_CHANGE_LAST_WRITE	= 0x00000010
FILE_NOTIFY_CHANGE_LAST_ACCESS	= 0x00000020
FILE_NOTIFY_CHANGE_CREATION		= 0x00000040
FILE_NOTIFY_CHANGE_SECURITY		= 0x00000100
FILE_ACTION_ADDED				= 0x00000001
FILE_ACTION_REMOVED				= 0x00000002
FILE_ACTION_MODIFIED			= 0x00000003
FILE_ACTION_RENAMED_OLD_NAME	= 0x00000004
FILE_ACTION_RENAMED_NEW_NAME	= 0x00000005
MAILSLOT_NO_MESSAGE				= -1
MAILSLOT_WAIT_FOREVER			= -1
FILE_CASE_SENSITIVE_SEARCH		= 0x00000001
FILE_CASE_PRESERVED_NAMES		= 0x00000002
FILE_UNICODE_ON_DISK			= 0x00000004
FILE_PERSISTENT_ACLS			= 0x00000008
FILE_FILE_COMPRESSION			= 0x00000010
FILE_VOLUME_IS_COMPRESSED		= 0x00008000

##
##  These are the generic rights.
##

GENERIC_READ					= 0x80000000
GENERIC_WRITE					= 0x40000000
GENERIC_EXECUTE					= 0x20000000
GENERIC_ALL						= 0x10000000

##
## File creation flags must start at the high end since they
## are combined with the attributes
##

FILE_FLAG_WRITE_THROUGH         = 0x80000000
FILE_FLAG_OVERLAPPED            = 0x40000000
FILE_FLAG_NO_BUFFERING          = 0x20000000
FILE_FLAG_RANDOM_ACCESS         = 0x10000000
FILE_FLAG_SEQUENTIAL_SCAN       = 0x08000000
FILE_FLAG_DELETE_ON_CLOSE       = 0x04000000
FILE_FLAG_BACKUP_SEMANTICS      = 0x02000000
FILE_FLAG_POSIX_SEMANTICS       = 0x01000000

CREATE_NEW          = 1
CREATE_ALWAYS       = 2
OPEN_EXISTING       = 3
OPEN_ALWAYS         = 4
TRUNCATE_EXISTING   = 5

##
## Define possible return codes from the CopyFileEx callback routine
##

PROGRESS_CONTINUE   = 0
PROGRESS_CANCEL     = 1
PROGRESS_STOP       = 2
PROGRESS_QUIET      = 3

##
## Define CopyFileEx callback routine state change values
##

CALLBACK_CHUNK_FINISHED         = 0x00000000
CALLBACK_STREAM_SWITCH          = 0x00000001

##
## Define CopyFileEx option flags
##

COPY_FILE_FAIL_IF_EXISTS        = 0x00000001
COPY_FILE_RESTARTABLE           = 0x00000002

PAGE_NOACCESS          = 0x01     
PAGE_READONLY          = 0x02     
PAGE_READWRITE         = 0x04     
PAGE_WRITECOPY         = 0x08     
PAGE_EXECUTE           = 0x10     
PAGE_EXECUTE_READ      = 0x20     
PAGE_EXECUTE_READWRITE = 0x40     
PAGE_EXECUTE_WRITECOPY = 0x80     
PAGE_GUARD            = 0x100     
PAGE_NOCACHE          = 0x200     
MEM_COMMIT           = 0x1000     
MEM_RESERVE          = 0x2000     
MEM_DECOMMIT         = 0x4000     
MEM_RELEASE          = 0x8000     
MEM_FREE            = 0x10000     
MEM_PRIVATE         = 0x20000     
MEM_MAPPED          = 0x40000     
MEM_RESET           = 0x80000     
MEM_TOP_DOWN       = 0x100000     
SEC_FILE           = 0x800000     
SEC_IMAGE         = 0x1000000     
SEC_RESERVE       = 0x4000000     
SEC_COMMIT        = 0x8000000     
SEC_NOCACHE      = 0x10000000     
MEM_IMAGE         = SEC_IMAGE     

SECTION_QUERY       = 0x0001
SECTION_MAP_WRITE   = 0x0002
SECTION_MAP_READ    = 0x0004
SECTION_MAP_EXECUTE = 0x0008
SECTION_EXTEND_SIZE = 0x0010
SECTION_ALL_ACCESS = (
	STANDARD_RIGHTS_REQUIRED
	| SECTION_QUERY
	| SECTION_MAP_WRITE
	| SECTION_MAP_READ
	| SECTION_MAP_EXECUTE
	| SECTION_EXTEND_SIZE
	)

FILE_MAP_COPY       = SECTION_QUERY
FILE_MAP_WRITE      = SECTION_MAP_WRITE
FILE_MAP_READ       = SECTION_MAP_READ
FILE_MAP_ALL_ACCESS = SECTION_ALL_ACCESS


# ===========================================================================


class file:
	def __init__ (self,
				  file_name,
				  desired_access		= GENERIC_READ,
				  share_mode			= FILE_SHARE_READ|FILE_SHARE_WRITE,
				  security_attributes	= 0,
				  creation_distribution	= OPEN_ALWAYS,
				  flags_and_attributes	= 0,
				  template_file			= 0
				  ):
		self.name = cstring (file_name)
		if security_attributes:
			raise ValueError, 'security attributes not yet supported'
		if template_file:
			raise ValueError, 'templates not yet supported'
		result = kernel32.CreateFile (
			self.name,
			desired_access,
			share_mode,
			security_attributes,
			creation_distribution,
			flags_and_attributes,
			template_file
			)
		if result == -1:
			raise error, "CreateFile: (%s)" % (winerror.get_error_string()[:-2])
		else:
			self.handle = result
		#
		self.overlapped_object = None
		
	def __int__ (self):
		return self.handle

	def close (self):
		if not kernel32.CloseHandle (self.handle):
			raise SystemError, "CloseHandle: (%s)" % (winerror.get_error_string()[:-2])
		else:
			self.handle = None

	def flush (self):
		if not kernel32.FlushFileBuffers (self.handle):
			raise SystemError, "FlushFileBuffers: (%s)" % (winerror.get_error_string()[:-2])

	def set_file_pointer (self, distance, how):
		result = kernel32.SetFilePointer (
			self.handle,
			distance,
			0,
			how
			)
		if result == -1:
			raise SystemError, "SetFilePointer: (%s)" % (winerror.get_error_string()[:-2])
		else:
			return result

	def get_file_size (self):
		return kernel32.GetFileSize (self.handle, 0)

	def __del__ (self):
		if self.handle:
			self.close()

	def overlapped (self):
		if not self.overlapped_object:
			self.overlapped_object = OVERLAPPED()
		return self.overlapped_object

	# can we use a bound method as the callback procedure?
	# that might simplify things.
	def async_write (self,
					 # a string, cstring, or membuf object
					 buffer,
					 # position in file
					 offset,
					 # callback object (a la gencb)
					 cb,
					 ):
		if type(buffer) == type(''):
			# pushing it here. 8^)
			addr = id(buffer)+12
			length = len(buffer)
		else:
			addr = buffer.address()
			length = buffer.size()
		self.overlapped().Offset = offset
		if not kernel32.WriteFileEx (self.handle, addr, length, self.overlapped(), cb):
			raise SystemError, "WriteFileEx: (%s)" % (winerror.get_error_string()[:-2])
		print '_after_ the call to WriteFileEx'
			
	def async_read (self,
					# cstring or membuf object
					buffer,
					# number of bytes to read
					num,
					# offset into file
					offset,
					# callback object
					cb
					):
		result = kernel32.ReadFileEx (
			self.handle,
			buffer.address(),
			num,
			self.overlapped(),
			cb
			)
		if not result:
			raise SystemError, "ReadFileEx: (%s)" % (winerror.get_error_string()[:-2])

import calldll

class memory_mapped_file:

	def __init__ (self, file):
		self.file = file # a win32 file object
		self.map()

	def __int__ (self):
		return self.address

	def get_membuf (self):
		size = self.file.get_file_size()
		return calldll.membuf (size, self.address)

	# TODO: make this take some arguments
	def map (self):
		self.map_handle = kernel32.CreateFileMapping (self.file, 0, PAGE_READONLY, 0, 0, 0)
		self.address = kernel32.MapViewOfFile (self.map_handle, FILE_MAP_READ, 0, 0, 0)
		
	def unmap (self):
		kernel32.UnmapViewOfFile (self.address)
		kernel32.CloseHandle (self.map_handle)
		self.address = self.map_handle = 0

	def __del__ (self):
		if self.address:
			self.unmap()

# ===========================================================================

import structob

class OVERLAPPED (structob.struct_object):
	oracle = structob.Oracle (
		'overlapped I/O object',
		'Nlllll',
		('Internal',
		 'InternalHigh',
		 'Offset',
		 'OffsetHigh',
		 'hEvent'
		 )
		)

if 0:
	def done_fun (*args):
		print 'callback: args %s' % repr(args)

	cb = gencb.callback ('lll', done_fun)

	def tf ():
		f = file (
			'test.txt',
			desired_access			= GENERIC_READ | GENERIC_WRITE,
			flags_and_attributes	= FILE_FLAG_OVERLAPPED
			)
		return f

	def go (f, pos = 1024 * 1024 * 10):
		f.async_write ('Hello There!', pos, cb)
		# enter an alertable wait state
		#kernel32.SleepEx (20000, 1)
