# -*- Mode: Python; tab-width: 4 -*-

import windll
import wintypes
import wingdi
import npstruct
import string

user32	= windll.module ('user32')
gdi32	= windll.module ('gdi32')

# ===========================================================================
#							   Constants
# ===========================================================================

# Should some of these constants belong to individual classes?
# for example, the mapping mode constants could conceivably
# be placed in the DC class definition.

# from winuser.h
# color types

CTLCOLOR_MSGBOX         = 0
CTLCOLOR_EDIT           = 1
CTLCOLOR_LISTBOX        = 2
CTLCOLOR_BTN            = 3
CTLCOLOR_DLG            = 4
CTLCOLOR_SCROLLBAR      = 5
CTLCOLOR_STATIC         = 6
CTLCOLOR_MAX            = 7

COLOR_SCROLLBAR         = 0
COLOR_BACKGROUND        = 1
COLOR_ACTIVECAPTION     = 2
COLOR_INACTIVECAPTION   = 3
COLOR_MENU              = 4
COLOR_WINDOW            = 5
COLOR_WINDOWFRAME       = 6
COLOR_MENUTEXT          = 7
COLOR_WINDOWTEXT        = 8
COLOR_CAPTIONTEXT       = 9
COLOR_ACTIVEBORDER      = 10
COLOR_INACTIVEBORDER    = 11
COLOR_APPWORKSPACE      = 12
COLOR_HIGHLIGHT         = 13
COLOR_HIGHLIGHTTEXT     = 14
COLOR_BTNFACE           = 15
COLOR_BTNSHADOW         = 16
COLOR_GRAYTEXT          = 17
COLOR_BTNTEXT           = 18
COLOR_INACTIVECAPTIONTEXT = 19
COLOR_BTNHIGHLIGHT      = 20

COLOR_3DDKSHADOW        = 21
COLOR_3DLIGHT           = 22
COLOR_INFOTEXT          = 23
COLOR_INFOBK            = 24
COLOR_DESKTOP           = COLOR_BACKGROUND

COLOR_3DFACE            = COLOR_BTNFACE
COLOR_3DSHADOW          = COLOR_BTNSHADOW
COLOR_3DHIGHLIGHT       = COLOR_BTNHIGHLIGHT
COLOR_3DHILIGHT         = COLOR_BTNHIGHLIGHT
COLOR_BTNHILIGHT        = COLOR_BTNHIGHLIGHT

#
# DrawText() Format Flags
#

DT_TOP              = 0x00000000
DT_LEFT             = 0x00000000
DT_CENTER           = 0x00000001
DT_RIGHT            = 0x00000002
DT_VCENTER          = 0x00000004
DT_BOTTOM           = 0x00000008
DT_WORDBREAK        = 0x00000010
DT_SINGLELINE       = 0x00000020
DT_EXPANDTABS       = 0x00000040
DT_TABSTOP          = 0x00000080
DT_NOCLIP           = 0x00000100
DT_EXTERNALLEADING  = 0x00000200
DT_CALCRECT         = 0x00000400
DT_NOPREFIX         = 0x00000800
DT_INTERNAL         = 0x00001000
DT_EDITCONTROL      = 0x00002000
DT_PATH_ELLIPSIS    = 0x00004000
DT_END_ELLIPSIS     = 0x00008000
DT_MODIFYSTRING     = 0x00010000
DT_RTLREADING       = 0x00020000
DT_WORD_ELLIPSIS    = 0x00040000


# ===========================================================================
# from wingdi.h
# ===========================================================================
# Device Parameters for GetDeviceCaps()

DRIVERVERSION	=	0		# Device driver version
TECHNOLOGY		=	2		# Device classification
HORZSIZE		=	4		# Horizontal size in millimeters
VERTSIZE		=	6		# Vertical size in millimeters
HORZRES			=	8		# Horizontal width in pixels
VERTRES			=	10		# Vertical height in pixels
BITSPIXEL		=	12		# Number of bits per pixel
PLANES			=	14		# Number of planes
NUMBRUSHES		=	16		# Number of brushes the device has
NUMPENS			=	18		# Number of pens the device has
NUMMARKERS		=	20		# Number of markers the device has
NUMFONTS		=	22		# Number of fonts the device has
NUMCOLORS		=	24		# Number of colors the device supports
PDEVICESIZE		=	26		# Size required for device descriptor
CURVECAPS		=	28		# Curve capabilities
LINECAPS		=	30		# Line capabilities
POLYGONALCAPS	=	32		# Polygonal capabilities
TEXTCAPS		=	34		# Text capabilities
CLIPCAPS		=	36		# Clipping capabilities
RASTERCAPS		=	38		# Bitblt capabilities
ASPECTX			=	40		# Length of the X leg
ASPECTY			=	42		# Length of the Y leg
ASPECTXY		=	44		# Length of the hypotenuse

LOGPIXELSX		=	88		# Logical pixels/inch in X
LOGPIXELSY		=	90		# Logical pixels/inch in Y

SIZEPALETTE		=	104		# Number of entries in physical palette
NUMRESERVED		=	106		# Number of reserved entries in palette
COLORRES		=	108		# Actual color resolution

# Printing related DeviceCaps. These replace the appropriate Escapes

PHYSICALWIDTH	=	110		# Physical Width in device units
PHYSICALHEIGHT	=	111		# Physical Height in device units 
PHYSICALOFFSETX	=	112		# Physical Printable Area x margin
PHYSICALOFFSETY	=	113		# Physical Printable Area y margin
SCALINGFACTORX	=	114		# Scaling factor x
SCALINGFACTORY	=	115		# Scaling factor y

# Display driver specific

VREFRESH		= 116	 	#	Current vertical refresh rate of the	
						 	#	display device (for displays only) in Hz
DESKTOPVERTRES	= 117	 	#	Horizontal width of entire desktop in	
						 	#	pixels									
DESKTOPHORZRES	= 118	 	#	Vertical height of entire desktop in	
						 	#	pixels									
BLTALIGNMENT	= 119	 	#	Preferred blt alignment					

#ifndef NOGDICAPMASKS

# Device Capability Masks:

# Device Technologies
DT_PLOTTER			= 0		#  Vector plotter					
DT_RASDISPLAY		= 1		#  Raster display					
DT_RASPRINTER		= 2		#  Raster printer					
DT_RASCAMERA		= 3		#  Raster camera					
DT_CHARSTREAM		= 4		#  Character-stream, PLP			
DT_METAFILE			= 5		#  Metafile, VDM					
DT_DISPFILE			= 6		#  Display-file						

#  Curve Capabilities 
CC_NONE				= 0		#  Curves not supported				
CC_CIRCLES			= 1		#  Can do circles					
CC_PIE				= 2		#  Can do pie wedges				
CC_CHORD			= 4		#  Can do chord arcs				
CC_ELLIPSES			= 8		#  Can do ellipese					
CC_WIDE				= 16	#  Can do wide lines				
CC_STYLED			= 32	#  Can do styled lines				
CC_WIDESTYLED		= 64	#  Can do wide styled lines			
CC_INTERIORS		= 128 	#  Can do interiors					
CC_ROUNDRECT		= 256 	#									

#  Line Capabilities 	
LC_NONE				= 0		#  Lines not supported				
LC_POLYLINE			= 2		#  Can do polylines					
LC_MARKER			= 4		#  Can do markers					
LC_POLYMARKER		= 8		#  Can do polymarkers				
LC_WIDE				= 16	#  Can do wide lines				
LC_STYLED			= 32	#  Can do styled lines				
LC_WIDESTYLED		= 64	#  Can do wide styled lines			
LC_INTERIORS		= 128 	#  Can do interiors					

#  Polygonal Capabilities 
PC_NONE				= 0		#  Polygonals not supported			
PC_POLYGON			= 1		#  Can do polygons					
PC_RECTANGLE		= 2		#  Can do rectangles				
PC_WINDPOLYGON		= 4		#  Can do winding polygons			
PC_TRAPEZOID		= 4		#  Can do trapezoids				
PC_SCANLINE			= 8		#  Can do scanlines					
PC_WIDE				= 16	#  Can do wide borders				
PC_STYLED			= 32	#  Can do styled borders			
PC_WIDESTYLED		= 64	#  Can do wide styled borders		
PC_INTERIORS		= 128 	#  Can do interiors					
PC_POLYPOLYGON		= 256 	#  Can do polypolygons				
PC_PATHS			= 512 	#  Can do paths						

#  Clipping Capabilities 
CP_NONE				= 0		#  No clipping of output			
CP_RECTANGLE		= 1		#  Output clipped to rects			
CP_REGION			= 2		#  obsolete							

#  Text Capabilities 
TC_OP_CHARACTER		= 0x00000001	#  Can do OutputPrecision	CHARACTER	   
TC_OP_STROKE		= 0x00000002	#  Can do OutputPrecision	STROKE		   
TC_CP_STROKE		= 0x00000004	#  Can do ClipPrecision		STROKE		   
TC_CR_90			= 0x00000008	#  Can do CharRotAbility	90			   
TC_CR_ANY			= 0x00000010	#  Can do CharRotAbility	ANY			   
TC_SF_X_YINDEP		= 0x00000020	#  Can do ScaleFreedom		X_YINDEPENDENT 
TC_SA_DOUBLE		= 0x00000040	#  Can do ScaleAbility		DOUBLE		   
TC_SA_INTEGER		= 0x00000080	#  Can do ScaleAbility		INTEGER		   
TC_SA_CONTIN		= 0x00000100	#  Can do ScaleAbility		CONTINUOUS	   
TC_EA_DOUBLE		= 0x00000200	#  Can do EmboldenAbility	DOUBLE		   
TC_IA_ABLE			= 0x00000400	#  Can do ItalisizeAbility	ABLE		   
TC_UA_ABLE			= 0x00000800	#  Can do UnderlineAbility	ABLE		   
TC_SO_ABLE			= 0x00001000	#  Can do StrikeOutAbility	ABLE		   
TC_RA_ABLE			= 0x00002000	#  Can do RasterFontAble	ABLE		   
TC_VA_ABLE			= 0x00004000	#  Can do VectorFontAble	ABLE		   
TC_RESERVED			= 0x00008000
TC_SCROLLBLT		= 0x00010000	#  Don't do text scroll with blt		   

#  Raster Capabilities 
RC_BITBLT			= 1				#  Can do standard BLT.				
RC_BANDING			= 2				#  Device requires banding support	
RC_SCALING			= 4				#  Device requires scaling support	
RC_BITMAP64			= 8				#  Device can support >64K bitmap	
RC_GDI20_OUTPUT		= 0x0010		#  has 2.0 output calls			
RC_GDI20_STATE		= 0x0020
RC_SAVEBITMAP		= 0x0040
RC_DI_BITMAP		= 0x0080		#  supports DIB to memory		
RC_PALETTE			= 0x0100		#  supports a palette			
RC_DIBTODEV			= 0x0200		#  supports DIBitsToDevice		
RC_BIGFONT			= 0x0400		#  supports >64K fonts			
RC_STRETCHBLT		= 0x0800		#  supports StretchBlt			
RC_FLOODFILL		= 0x1000		#  supports FloodFill			
RC_STRETCHDIB		= 0x2000		#  supports StretchDIBits		
RC_OP_DX_OUTPUT		= 0x4000
RC_DEVBITS			= 0x8000


#  Background Modes 
TRANSPARENT         = 1
OPAQUE              = 2
BKMODE_LAST         = 2

#  Graphics Modes 

GM_COMPATIBLE       = 1
GM_ADVANCED         = 2
GM_LAST             = 2

#  Mapping Modes 
MM_TEXT             = 1
MM_LOMETRIC         = 2
MM_HIMETRIC         = 3
MM_LOENGLISH        = 4
MM_HIENGLISH        = 5
MM_TWIPS            = 6
MM_ISOTROPIC        = 7
MM_ANISOTROPIC      = 8

#  CombineRgn() Styles 
RGN_AND             = 1
RGN_OR              = 2
RGN_XOR             = 3
RGN_DIFF            = 4
RGN_COPY            = 5
RGN_MIN             = RGN_AND
RGN_MAX             = RGN_COPY

#  3D border styles 
BDR_RAISEDOUTER = 0x0001
BDR_SUNKENOUTER = 0x0002
BDR_RAISEDINNER = 0x0004
BDR_SUNKENINNER = 0x0008

BDR_OUTER       = 0x0003
BDR_INNER       = 0x000c
BDR_RAISED      = 0x0005
BDR_SUNKEN      = 0x000a


EDGE_RAISED     = (BDR_RAISEDOUTER | BDR_RAISEDINNER)
EDGE_SUNKEN     = (BDR_SUNKENOUTER | BDR_SUNKENINNER)
EDGE_ETCHED     = (BDR_SUNKENOUTER | BDR_RAISEDINNER)
EDGE_BUMP       = (BDR_RAISEDOUTER | BDR_SUNKENINNER)

#  Border flags 
BF_LEFT         = 0x0001
BF_TOP          = 0x0002
BF_RIGHT        = 0x0004
BF_BOTTOM       = 0x0008

BF_TOPLEFT      = (BF_TOP | BF_LEFT)
BF_TOPRIGHT     = (BF_TOP | BF_RIGHT)
BF_BOTTOMLEFT   = (BF_BOTTOM | BF_LEFT)
BF_BOTTOMRIGHT  = (BF_BOTTOM | BF_RIGHT)
BF_RECT         = (BF_LEFT | BF_TOP | BF_RIGHT | BF_BOTTOM)

BF_DIAGONAL     = 0x0010

# For diagonal lines, the BF_RECT flags specify the end point of the
# vector bounded by the rectangle parameter.
BF_DIAGONAL_ENDTOPRIGHT     = (BF_DIAGONAL | BF_TOP | BF_RIGHT)
BF_DIAGONAL_ENDTOPLEFT      = (BF_DIAGONAL | BF_TOP | BF_LEFT)
BF_DIAGONAL_ENDBOTTOMLEFT   = (BF_DIAGONAL | BF_BOTTOM | BF_LEFT)
BF_DIAGONAL_ENDBOTTOMRIGHT  = (BF_DIAGONAL | BF_BOTTOM | BF_RIGHT)


BF_MIDDLE       = 0x0800  #  Fill in the middle 
BF_SOFT         = 0x1000  #  For softer buttons 
BF_ADJUST       = 0x2000  #  Calculate the space left over 
BF_FLAT         = 0x4000  #  For flat rather than 3D borders 
BF_MONO         = 0x8000  #  For monochrome borders 

ETO_OPAQUE                   = 0x0002
ETO_CLIPPED                  = 0x0004
ETO_GLYPH_INDEX              = 0x0010
ETO_RTLREADING               = 0x0080
ETO_IGNORELANGUAGE           = 0x1000

# ===========================================================================
#							   Utility
# ===========================================================================

# TODO: we should really build a color object, with __int__ conversion.

def pack_color ((r,g,b)):
	return  ((b&0xff) << 16) | ((g&0xff) << 8) | (r&0xff)

def unpack_color (n):
	return (n&0xff), (n&0xff00) >> 8, (n&0xff0000) >> 16

def get_sys_color (which):
	return unpack_color (user32.GetSysColor (which))

def get_sys_pen (color):
	r,g,b = get_sys_color (color)
	return wingdi.pen (red=r,green=g, blue=b).create()

def get_sys_brush (color):
	r,g,b = get_sys_color (color)
	return wingdi.brush (red=r,green=g, blue=b).create()

# ===========================================================================
#							   Objects
# ===========================================================================

class DC:
	def __init__ (self, handle):
		self.handle = handle

	def get_handle (self):
		return self.handle

	__int__ = get_handle

	def create_compatible_dc (self):
		return self.__class__ (
			gdi32.CreateCompatibleDC (self)
			)

	def create_compatible_bitmap (self, width, height):
		handle = gdi32.CreateCompatibleBitmap (self, width, height)
		if handle:
			# wingdi needs a mechanism for creating a specific
			# GDI object from a handle.
			return wingdi.GDIOBJ (handle)
		else:
			return handle
	
	def delete_dc (self):
		h, self.handle = self.handle, 0
		return gdi32.DeleteDC (h)

	def release_dc (self):
		h, self.handle = self.handle, 0
		return user32.ReleaseDC (h)

	# ==================================================
	#  printing
	# ==================================================
	
	def start_doc (self, docinfo):
		return gdi32.StartDoc (self, docinfo)
		
	def start_page (self):
		return gdi32.StartPage (self)

	def end_page (self):
		return gdi32.EndPage (self)

	def end_doc (self):
		return gdi32.EndDoc (self)

	# ==================================================
	#  properties
	# ==================================================

	def set_bk_color (self, color=None):
		if color is None:
			color = user32.GetSysColor (COLOR_WINDOW)
		if type(color) == type(()):
			color = pack_color (color)
		return gdi32.SetBkColor (self, color)
		
	def set_bk_mode (self, mode=TRANSPARENT):
		return gdi32.SetBkMode (self, mode)

	def set_text_align (self, mode):
		return gdi32.SetTextAlign (self, mode)

	def set_text_color (self, color):
		return gdi32.SetTextColor (self, pack_color (color))

	def get_text_color (self):
		return unpack_color (gdi32.GetTextColor())

	def get_text_extent (self, text):
		size = wintypes.SIZE()
		ctext = windll.cstring (text)
		gdi32.GetTextExtentPoint32 (
			self,
			ctext,
			len(text),
			size
			)
		return size.x, size.y

	def get_char_width (self, low, high):
		n = (high-low)+1
		mb = windll.membuf (n*4)
		result = gdi32.GetCharWidth32 (
			self.handle,
			low,
			high,
			mb
			)
		if result:
			return npstruct.unpack ('l'*n, mb.read())[0]
		else:
			raise SystemError, "gdi32.GetCharWidth32() failed"

	def get_text_metrics (self):
		import winfont
		TM = winfont.NEWTEXTMETRIC()
		gdi32.GetTextMetrics (self, TM)
		return TM

	def select_object (self, gdi_object):
		old_handle = gdi32.SelectObject (
			self,
			gdi_object
			)
		if wingdi.GDIOBJ.gdi_obj_map.has_key (old_handle):
			return wingdi.GDIOBJ.gdi_obj_map[old_handle]
		else:
			return old_handle

	# ==================================================
	#  graphics
	# ==================================================

	def text_out (self, (x, y), txt):
		# assume it's a cstring if it's not a python string
		if type(txt) == type(''):
			txt = windll.cstring (txt)
		return gdi32.TextOut (self, x, y, txt, len(txt))

	def draw_text (self, text, rect, flags=DT_CENTER|DT_WORDBREAK):
		r = wintypes.t_rect()
		r.rect = rect
		text = windll.cstring (text)
		user32.DrawText (
			self.handle,
			text,
			len(text),
			r,
			flags
			)

	def draw_focus_rect (self, rect):
		r = wintypes.t_rect()
		r.rect = rect
		user32.DrawFocusRect (self.handle, r)

	def move_to (self, (x, y)):
		return gdi32.MoveToEx (self.handle, x, y, 0)

	def move_to_ex (self, (x, y)):
		P = wintypes.POINT()
		gdi32.MoveToEx (self.handle, x, y, P)
		return P.x, P.y

	def line_to (self, (x, y)):
		return gdi32.LineTo (self.handle, x, y)

	# a pre-allocated RECT buffer as a tuple
	scratch_rect = wintypes.t_rect()

	# need to make use of scratch_rect optional
	def fill_rect (self, (l,t,r,b), brush):
		user32.SetRect (self.scratch_rect, l, t, r, b)
		return user32.FillRect (self, self.scratch_rect, brush)

	# This is an MFC thing, supposed to be much faster than FillRect,
	# but there's probably no noticable speed difference to us...
	def fill_solid_rect (self, rect, color=COLOR_WINDOW):
		save = self.set_bk_color (color)
		r = wintypes.t_rect()
		r.rect = rect
		gdi32.ExtTextOut (self.handle, 0, 0, ETO_OPAQUE, r, 0, 0, 0)
		#self.set_bk_color (save)

	def rectangle (self, (l,t,r,b)):
		return gdi32.Rectangle (self, l, t, r, b)

	def round_rect (self, (l,t,r,b),w,h):
		return gdi32.RoundRect (self, l, t, r, b, w, h)

	def frame_rect (self, (l,t,r,b), brush):
		user32.SetRect (self.scratch_rect, l, t, r, b)		
		return user32.FrameRect (self, self.scratch_rect, brush)		

	def get_device_caps (self, cap):
		return gdi32.GetDeviceCaps (self, cap)

	def get_clip_box (self):
		r = wintypes.t_rect()
		gdi32.GetClipBox (self, r)
		return r.rect

	def ellipse (self, (l, t, r, b)):
		return gdi32.Ellipse (self, l, t, r, b)

	def arc (self, (l, t, r, b), (asx, asy), (aex, aey)):
		return gdi32.Arc (self, l, t, r, b, asx, asy, aex, aey)
		
	def pie (self, (l, t, r, b), (asx, asy), (aex, aey)):
		return gdi32.Pie (self, l, t, r, b, asx, asy, aex, aey)

	def bit_blt (self, (dx, dy), (w, h), source_dc, (sx, sy), rop):
		return gdi32.BitBlt (
			self.handle,
			dx, dy,
			w, h,
			source_dc,
			sx, sy,
			rop
			)

	def draw_edge (self, (l,t,r,b), edge=EDGE_RAISED, flags=BF_RECT):
		rect = wintypes.t_rect ()
		user32.SetRect (rect, l, t, r, b)
		return user32.DrawEdge (self, rect, edge, flags)

	def draw_3d_box (self, rect, up=1):
		l,t,r,b = rect
		# draw face rect
		self.fill_rect (rect, get_sys_brush (COLOR_3DFACE))

		if up is None:
			# kludge, up=None just draws the rectangle
			return

		# draw dark shadow
		pd = get_sys_pen (COLOR_3DDKSHADOW) 
		ps = get_sys_pen (COLOR_3DSHADOW)
		ph = get_sys_pen (COLOR_3DHILIGHT)
		pf = get_sys_pen (COLOR_3DFACE)

		if not up:
			pd,ps,ph = ph,pf,ps

		self.select_object (pd)
		self.move_to ((r,t))
		self.line_to ((r,b))
		self.line_to ((l-1,b))
		# draw shadow
		self.select_object (ps)
		self.move_to ((r-1,t+1))
		self.line_to ((r-1,b-1))
		self.line_to ((l,b-1))
		# draw hilight
		self.select_object (ph)
		self.move_to ((l,b-1))
		self.line_to ((l,t))
		self.line_to ((r,t))

	# alternative implementation based on DrawEdge
	def draw_3d_box (self, rect, kind='raised', filled=1):
		if filled:
			self.fill_rect (rect, get_sys_brush (COLOR_3DFACE))

		if kind is None:
			# kludge, kind=None just draws the rectangle
			return

		if type(kind) == type(0):
			# compatibility
			if kind:
				kind = 'raised'
			else:
				kind = 'sunken'
			
		flags = {
			'etched': EDGE_ETCHED,
			'raised': EDGE_RAISED,
			'sunken': EDGE_SUNKEN,
			'bump'  : EDGE_BUMP
			}[kind]

		self.draw_edge (rect, flags)

	def polygon (self, *points):
		data = self._pack_point_array (points)
		return gdi32.Polygon (self, data, len(points))

	# ==================================================
	#  paths
	# ==================================================

	def begin_path (self):
		return gdi32.BeginPath (self)

	def end_path (self):
		return gdi32.EndPath (self)

	def select_clip_path (self, mode=RGN_COPY):
		return gdi32.SelectClipPath (self, mode)

	def select_clip_rgn (self, region=0):
		return gdi32.SelectClipRgn (self, region)

	def stroke_path (self):
		return gdi32.StrokePath (self)

	def fill_path (self):
		return gdi32.FillPath (self)

	def stroke_and_fill_path (self):
		return gdi32.StrokeAndFillPath (self)		

	# ==================================================
	# logical coordinate space transformations
	# ==================================================

	# When using MM_ISOTROPIC, it is important to call
	# set_window_ext _BEFORE_ calling set_viewport_ext

	def set_map_mode (self, mode):
		return gdi32.SetMapMode (self, mode)

	# ----------
	# converting between device and logical points
	# ----------

	def _pack_point_array (self, points):
		n = len(points)
		data = string.join (map (lambda p: npstruct.pack ('Nll', p), points), '')
		mb = windll.membuf (len (data))
		mb.write (data)
		return mb

	def _unpack_point_array (self, buffer):
		data = buffer.read()
		n = len (data) / 8
		r = []
		for i in range (n):
			m = i * 4
			r.append (npstruct.unpack ('Nll', data[m:m+8])[0])
		if len (r) == 1:
			# the most common case, make it obvious
			return r[0]
		else:
			return tuple (r)

	def dp_to_lp (self, *points):
		data = self._pack_point_array (points)
		gdi32.DPtoLP (self, data, len(points))
		return self._unpack_point_array (data)

	def lp_to_dp (self, *points):
		data = self._pack_point_array (points)
		gdi32.LPtoDP (self, data, len(points))
		return self._unpack_point_array (data)

	# ----------
	# window
	# ----------

	# scratch point buffer
	_logical_point_scratch = wintypes.POINT()

	def set_window_org (self, x, y):
		# we ignore the point parameter
		return gdi32.SetWindowOrgEx (self, x, y, 0)

	def get_window_org (self):
		p = self._logical_point_scratch
		result = gdi32.GetWindowOrgEx (self, p)
		if result:
			return p.x, p.y
		else:
			raise SystemError, "GetWindowOrgEx() failed"

	def set_window_ext (self, w, h):
		return gdi32.SetWindowExtEx (self, w, h, 0)

	def get_window_ext (self):
		p = self._logical_point_scratch
		result = gdi32.GetWindowExtEx (self, p)
		if result:
			return p.x, p.y
		else:
			raise SystemError, "GetWindowExtEx() failed"

	# ----------
	# viewport
	# ----------

	def set_viewport_org (self, x, y):
		# we ignore the point parameter
		return gdi32.SetViewportOrgEx (self, x, y, 0)

	def get_viewport_org (self):
		p = self._logical_point_scratch
		result = gdi32.GetViewportOrgEx (self, p)
		if result:
			return p.x, p.y
		else:
			raise SystemError, "GetViewportOrgEx() failed"

	def set_viewport_ext (self, w, h):
		return gdi32.SetViewportExtEx (self, w, h, 0)

	def get_viewport_ext (self):
		p = self._logical_point_scratch
		result = gdi32.GetViewportExtEx (self, p)
		if result:
			return p.x, p.y
		else:
			raise SystemError, "GetViewportExtEx() failed"
