# -*- Mode: Python; tab-width: 4 -*-
#	Author: Sam Rushing <rushing@nightmare.com>

# support for the new extended menu capability.
# [under construction]

import npstruct

#from dyn_win32 import windll, structob
import windll
import structob

user32 = windll.module ('user32')

class MENUITEMINFO (structob.struct_object):
	oracle = structob.Oracle (
		'menu item info',
		'Nlllllllllll',
		('cbSize',
		 'fMask',
		 'fType',
		 'fState',
		 'wID',
		 'hSubMenu',
		 'hbmpChecked',
		 'hbmpUnchecked',
		 'dwItemData',
		 'dwTypeData',
		 'cch')
		)

# values for fMask
# Retrieves or sets the fState member.
MIIM_STATE		= 0x00000001
# Retrieves or sets the wID member.
MIIM_ID			= 0x00000002
# Retrieves or sets the hSubMenu member.
MIIM_SUBMENU	= 0x00000004
# Retrieves or sets the hbmpChecked and hbmpUnchecked members
MIIM_CHECKMARKS	= 0x00000008
# Retrieves or sets the fType and dwTypeData members.
MIIM_TYPE		= 0x00000010
# Retrieves or sets the dwItemData member.
MIIM_DATA		= 0x00000020

# values for fType
#
# MFT_BITMAP Displays the menu item using a bitmap. The low-order word
# of the dwTypeData member is the bitmap handle, and the cch member is
# ignored.
#
# MFT_MENUBARBREAK Places the menu item on a new line if it is in a
# menu bar or in a new column if it is in a pop-up menu. A vertical
# line separates the new column from the old.
#
# MFT_MENUBREAK Places the menu item on a new line if it is in a menu
# bar or in a new column if it is in a pop-up menu. The columns are not
# separated by a vertical line.	 MFT_OWNERDRAW Assigns responsibility
# for drawing the menu item to the window that owns the menu. The window
# receives a WM_MEASUREITEM message before the menu is displayed for the
# first time, and a WM_DRAWITEM message whenever the appearance of the
# menu item must be updated. If this value is specified, the dwTypeData
# member contains an application-defined 32-bit value.
# 
# MFT_RADIOCHECK Displays checked menu items using a radio-button mark
# instead of a check mark if the hbmpChecked member is NULL.
# 
# MFT_RIGHTJUSTIFY Right-justifies the menu item and any subsequent
# items. This value is valid only if the menu item is in a menu bar.
# 
# MFT_SEPARATOR Specifies that the menu item is a separator. A menu item
# separator appears as a horizontal dividing line. The dwTypeData and
# cch members are ignored. This value is valid only if the menu item is
# in a pop-up menu.
# 
# MFT_STRING Displays the menu item using a text string. The dwTypeData
# member is the pointer to a null-terminated string, and the cch member
# is the length of the string.
# 
# The MFT_BITMAP, MFT_SEPARATOR, and MFT_STRING values cannot be
# combined with one another.

MFT_STRING			= 0x00000000
MFT_BITMAP			= 0x00000004
MFT_MENUBARBREAK	= 0x00000020
MFT_MENUBREAK		= 0x00000040
MFT_OWNERDRAW		= 0x00000100
MFT_RADIOCHECK		= 0x00000200
MFT_SEPARATOR		= 0x00000800
MFT_RIGHTORDER		= 0x00002000
MFT_RIGHTJUSTIFY	= 0x00004000

# values for fState
# MFS_CHECKED Checks the menu item. For more information about checked
#	menu items, see the hbmpChecked member.
# MFS_DEFAULT Specifies that the menu item is the default. A menu can
#	contain only one default menu item, which is displayed in bold.
# MFS_DISABLED Disables the menu item so that it cannot be selected,
#	but does not gray it.
# MFS_ENABLED Enables the menu item so that it can be selected. This
#	is the default state.
# MFS_GRAYED Disables the menu item and grays it so that it cannot be
#	selected.
# MFS_HILITE	Highlights the menu item.
# MFS_UNCHECKED Unchecks the menu item. For more information about
#	unchecked menu items, see the hbmpUnchecked member.
# MFS_UNHILITE Removes the highlight from the menu item. This is the
#	default state.

MFS_GRAYED	=	0x00000003
MFS_DISABLED=	MFS_GRAYED	# conflicts with the above definition
MFS_CHECKED	=	0x00000008
MFS_HILITE	=	0x00000080
MFS_ENABLED	=	0x00000000
MFS_DEFAULT	=	0x00001000
# the corresponding MFS_UNXXX are just zero.

class menu_item:
	def __init__ (self, id):
		mii	= MENUITEMINFO()
		mii.cbSize = mii._size
		mii.fMask = 0
		mii.fType = 0
		mii.fState = 0
		mii.wID = 0
		mii.hSubMenu = 0
		mii.hbmpChecked = 0
		mii.hbmpUnchecked = 0
		mii.dwItemData = id
		mii.dwTypeData = 0
		mii.cch = 0
		self.mii = mii

	def address (self):
		return self.mii.address()

	def enable (self, yesno=1):
		if yesno:
			self.mii.fState = self.mii.fState | MFS_GRAYED
		else:
			self.mii.fState = self.mii.fState & ~MFS_GRAYED

	def check (self, yesno=1):
		if yesno:
			self.mii.fState = self.mii.fState | MFS_CHECKED
		else:
			self.mii.fState = self.mii.fState & ~MFS_CHECKED

class text_menu_item (menu_item):
	def __init__ (self, id, label, enabled=1):
		menu_item.__init__ (self, id)
		self.label = windll.cstring (label)
		self.mii.fType = MFT_STRING
		if enabled:
			self.mii.fState = MFS_ENABLED
		else:
			self.mii.fState = MFS_GRAYED
		self.mii.dwTypeData = self.label.address()
		self.mii.cch = len(label)

class separator_menu_item (menu_item):
	def __init__ (self):
		menu_item.__init__ (self, 0)
		self.mii.fType = MFT_SEPARATOR

class menu:
	def __init__ (self, items=None):
		self.hmenu = 0
		if items == None:
			self.items = []
		else:
			self.items = items

	def __del__ (self):
		if self.hmenu:
			self.destroy()

	def destroy (self):
		user32.DestroyMenu (self.hmenu)		

	def create (self):
		self.hmenu = user32.CreateMenu ()

	def set_items (self, items):
		# discard the old menu and items
		self.destroy()
		# start anew...
		self.create()
		self.items = items
		for i in range(len(items)):
			user32.InsertMenuItem (self.hmenu,
								i+1,
								1,
								items[i].address())

	def attach (self, window):
		if self.hmenu:
			user32.SetMenu (window.hwnd, self.hmenu)
		else:
			raise ValueError, "menu not instantiated"

class popup_menu (menu):
	def create (self):
		self.hmenu = user32.CreatePopupMenu ()
	

def demo ():
	#mb = menu()
	#mb.set_items ([text_menu_item (44, 'top level')])
	sub_items = [text_menu_item (42, 'first item'),
				 separator_menu_item (),
				 text_menu_item (43, 'second item')]
	pm = popup_menu()
	pm.set_items (sub_items)
	return pm
	
	
