# -*- Mode: Python; tab-width: 4 -*-

#from dyn_win32 import windll, winwin, structob
import windll
import winwin
import structob

cstring = windll.cstring

kernel32 = windll.module ('kernel32')
user32	 = windll.module ('user32')
gdi32	 = windll.module ('gdi32')

# ===========================================================================
# Scrollable Windows
# ===========================================================================

class SCROLLINFO (structob.struct_object):
	oracle = structob.Oracle (
		'scroll info',
		'Nlllllll',
		('size',
		 'mask',
		 'min',
		 'max',
		 'page',
		 'pos',
		 'track_pos'
		 )
		)

# There are several new capabilities with Windows 4.0, and the
# interface has been simplified... most everything can be done
# through {Get,Set}ScrollInfo.  One big win is that we can now
# process 32-bit scroll positions.
#
# SetScrollInfo
# GetScrollInfo
#
# [from the win32 docs]
#   In version 4.0 or later, the maximum value that a scroll bar can
#   report (that is, the maximum scrolling position) depends on the
#   page size. If the scroll bar has a page size greater than one, the
#   maximum scrolling position is less than the maximum range
#   value. You can use the following formula to calculate the maximum
#   scrolling position:
#   MaxScrollPos = MaxRangeValue - (PageSize - 1) 

# Scroll Bar Constants

SB_HORZ             = 0
SB_VERT             = 1
SB_CTL              = 2
SB_BOTH             = 3

# Scroll Bar Commands

SB_LINEUP           = 0
SB_LINELEFT         = 0
SB_LINEDOWN         = 1
SB_LINERIGHT        = 1
SB_PAGEUP           = 2
SB_PAGELEFT         = 2
SB_PAGEDOWN         = 3
SB_PAGERIGHT        = 3
SB_THUMBPOSITION    = 4
SB_THUMBTRACK       = 5
SB_TOP              = 6
SB_LEFT             = 6
SB_BOTTOM           = 7
SB_RIGHT            = 7
SB_ENDSCROLL        = 8

# Scroll Bar Styles
SBS_HORZ					= 0x0000L
SBS_VERT					= 0x0001L
SBS_TOPALIGN				= 0x0002L
SBS_LEFTALIGN				= 0x0002L
SBS_BOTTOMALIGN				= 0x0004L
SBS_RIGHTALIGN				= 0x0004L
SBS_SIZEBOXTOPLEFTALIGN		= 0x0002L
SBS_SIZEBOXBOTTOMRIGHTALIGN = 0x0004L
SBS_SIZEBOX					= 0x0008L
SBS_SIZEGRIP				= 0x0010L

# Scroll bar messages

SBM_SETPOS					= 0x00E0 # not in win3.1 
SBM_GETPOS					= 0x00E1 # not in win3.1 
SBM_SETRANGE				= 0x00E2 # not in win3.1 
SBM_SETRANGEREDRAW			= 0x00E6 # not in win3.1 
SBM_GETRANGE				= 0x00E3 # not in win3.1 
SBM_ENABLE_ARROWS			= 0x00E4 # not in win3.1 
SBM_SETSCROLLINFO			= 0x00E9
SBM_GETSCROLLINFO			= 0x00EA

SIF_RANGE			= 0x0001
SIF_PAGE			= 0x0002
SIF_POS				= 0x0004
SIF_DISABLENOSCROLL = 0x0008
SIF_TRACKPOS		= 0x0010
SIF_ALL				= (SIF_RANGE | SIF_PAGE | SIF_POS | SIF_TRACKPOS)

SPLIT_WORD = winwin.SPLIT_WORD

# Scrolling a window is done via the manipulation of the logical
# coordinate space associated with the DC. [as such, the interfaces
# probably belong in windc].
#
# SetWindowOrg
# SetWindowExt
#

# Traditional scrolling is either X or Y.  It is possible
# to scroll both, however, and have only the relevant areas
# moved and updated. [like with a little 'drag-box' scroller]

# trace of scroll event steps:
# 1) click one line down
# 2) move the current y position
# 3) scroll the window
# 4) update the window

# Right after a BeginPaint(), we set the logical coordinates
# for the DC before painting anything.  All paint-type ops
# are w.r.t. logical space, not device space.

# think about 'mixins', i.e., 'scrollable_window_mixin'.

class scrollable_window (winwin.python_window):
	scroll_info = SCROLLINFO()

	def create (self):
		hwnd = winwin.python_window.create (self)
		self.style  = self.style | winwin.WS_VSCROLL | winwin.WS_HSCROLL
		# NOTE: This is very important!  If you don't set size,
		# then win32 will assume that there's no nTrackPos slot,
		# and will silently ignore anything you try to do with it!
		self.scroll_info.size = self.scroll_info._size
		self.x_pos = self.y_pos = 0
		self.width = self.height = 512
		self.set_virtual_window ((0,0,2048,2048))
		return hwnd

	def command_name (self, pos):
		return ['SB_LINE{UP,LEFT}', 'SB_LINE{DOWN,RIGHT}',
				'SB_PAGE{UP,LEFT}', 'SB_PAGE{DOWN,RIGHT}',
				'SB_THUMBPOSITION', 'SB_THUMBTRACK',
				'SB_{TOP,LEFT}', 'SB_{BOTTOM,RIGHT}',
				'SB_ENDSCROLL'
				][pos]

	def get_scroll_info (self, which):
		si = self.scroll_info
		si.mask = SIF_ALL
		result = user32.GetScrollInfo (self, which, si)
		if result:
			return si.min, si.max, si.page, si.pos, si.track_pos
		else:
			raise SystemError, 'GetScrollInfo() failed'

	def set_scroll_info (self, which, **info):
		si = self.scroll_info
		for k,v in info:
			setattr (si,k,v)
		return user32.SetScrollInfo (self, which, si)

	# ======================================================================
	# Message Handlers
	# ======================================================================

	def WM_SIZE (self, wparam, lparam):
		w, h = SPLIT_WORD (lparam)
		self.width = w
		self.height = h
		self.set_scroll_values()

	def get_logical_height (self):
		return self.height

	def get_logical_width (self):
		return self.width

	# scroll position information is encoded in the HIWORD of wparam,
	# but it is only 16 bits.  we want the full 32-bit range, so we
	# use the GetScrollInfo() interface.

	def WM_HSCROLL (self, wparam, lparam):
		scroll_code = SPLIT_WORD (wparam) [0]

		si = self.scroll_info
		si.mask = SIF_ALL
		user32.GetScrollInfo (self, SB_HORZ, si)
		
		x_pos = old_x_pos = si.pos

		l,t,r,b = self.get_virtual_window()
		x_line, y_line, x_page, y_page = self.get_scroll_values()

		dx = 0
		if scroll_code == SB_THUMBTRACK:
			dx = si.track_pos - x_pos
		elif scroll_code == SB_LINEUP:
			dx = - x_line
		elif scroll_code == SB_PAGEUP:
			dx = - x_page
		elif scroll_code == SB_LINEDOWN:
			dx = x_line
		elif scroll_code == SB_PAGEDOWN:
			dx = x_page

		x_pos = x_pos + dx

		# range checking
		x_pos = max (min (x_pos, r - self.get_logical_width()), l)

		ldx = old_x_pos - x_pos

		if not ldx:
			return

		# set the new position
		si.mask = SIF_POS
		si.pos = self.x_pos = x_pos
		user32.SetScrollInfo (self, SB_HORZ, si, 1)

		# scroll the window's contents
		self.scroll_window (ldx, 0)

		return 0

	def WM_VSCROLL (self, wparam, lparam):
		scroll_code = SPLIT_WORD (wparam)[0]

		si = self.scroll_info
		si.mask = SIF_ALL
		user32.GetScrollInfo (self, SB_VERT, si)
		
		y_pos = old_y_pos = si.pos

		l,t,r,b = self.get_virtual_window()
		x_line, y_line, x_page, y_page = self.get_scroll_values()

		dy = 0
		if scroll_code == SB_THUMBTRACK:
			dy = si.track_pos - y_pos
		elif scroll_code == SB_LINEUP:
			dy = y_line
		elif scroll_code == SB_PAGEUP:
			dy = - y_page
		elif scroll_code == SB_LINEDOWN:
			dy = - y_line
		elif scroll_code == SB_PAGEDOWN:
			dy = + y_page

		y_pos = y_pos + dy
		
		# range checking
		y_pos = max (min (y_pos, b - self.get_logical_height()), t)

		ldy = old_y_pos - y_pos

		if not ldy:
			return 0

		# set the new position
		si.mask = SIF_POS
		si.pos = self.y_pos = y_pos
		user32.SetScrollInfo (self, SB_VERT, si, 1)

		# scroll the window's contents
		self.scroll_window (0, ldy)

		return 0

	def begin_paint (self, paintstruct=None):
		dc = winwin.python_window.begin_paint (self, paintstruct)
		self.prepare_dc (dc)
		return dc

	def prepare_dc (self, dc):
		dc.set_window_org (self.x_pos, self.y_pos)

	# ======================================================================
	# Interface
	# ======================================================================

	# set the boundaries of the virtual window.

	def set_virtual_window (self, rect):
		self.virtual_window = rect
		# this should only be necessary if it's changed after the
		# window's already been created?
		self.set_scroll_values()

	def get_virtual_window (self):
		return self.virtual_window

	# ==============================

	def set_scroll_values (self, x_line=None, y_line=None, x_page=None, y_page=None):
		vl,vt,vr,vb = self.get_virtual_window()
		vw, vh = vr - vl, vt - vb

		# these are most likely what you will want to override.
		# often these are based on a font height.
		if x_line is None:
			x_line = vw/100
		if y_line is None:
			y_line = vh/100

		# reasonable defaults for page sizes
		if x_page is None:
			x_page = self.get_logical_width()
		if y_page is None:
			y_page = self.get_logical_height()

		self.scroll_values = (x_line, y_line, x_page, y_page)
		self.scroll_range_check()
		self.set_scroll_info()

	def scroll_range_check (self):
		# range checking on current scroll position.  we need only
		# adjust position when a dimension has collapsed to smaller
		# than the current physical window size.

		l,t,r,b = self.get_virtual_window()
		
		if self.width > (r-l):
			if self.x_pos != l:
				self.x_pos = l
				self.invalidate_rect (0)

		if self.height > (b-t):
			if self.y_pos != t:
				self.y_pos = t
				self.invalidate_rect (0)

	def get_scroll_values (self):
		return self.scroll_values

	# ==============================

	def set_scroll_info (self):
		vl,vt,vr,vb = self.get_virtual_window()
		vw, vh = vr - vl, vt - vb

		x_line, y_line, x_page, y_page = self.get_scroll_values()

		si = self.scroll_info

		# set the vertical scroll info

		si.min = vt
		si.max = vb
		si.page = y_page
		si.pos = self.y_pos
		si.mask = SIF_PAGE | SIF_POS | SIF_RANGE
		if self.hwnd:
			user32.SetScrollInfo (self, SB_VERT, si)

		# set the horizontal scroll info

		si.min = vl
		si.max = vr
		si.page = x_page
		si.pos = self.x_pos
		si.mask = SIF_PAGE | SIF_POS | SIF_RANGE

		if self.hwnd:
			user32.SetScrollInfo (self, SB_HORZ, si)


	def scroll_window (self,
					   dx, dy,
					   scroll_rect=0,
					   clip_rect=0):
		# it is tempting to use ScrollWindowEx here, but
		# I think it requires specifying an explicit update
		# region, kindofa pain.
		return user32.ScrollWindow (
			self,
			dx, dy,
			scroll_rect,
			clip_rect
			)

	# ==============================
	# any obvious use of this will have rect in logical coordinates.
	#
	def invalidate_rect (self, rect, erase=1):
		if rect:
			x, y = self.x_pos, self.y_pos
			l,t,r,b = rect
			rect = l-x, t-y, r-x, b-y

		winwin.python_window.invalidate_rect (self, rect, erase)

	def dp_to_lp (self, (x,y)):
		return (x+self.x_pos, y+self.y_pos)

	def lp_to_dp (self, (x,y)):
		return (x-self.x_pos, y-self.y_pos)
