# -*- Mode: Python; tab-width: 4 -*-

#from dyn_win32 import windll, windc, wintypes, gencb, winmesg, winclass
import windll
import windc
import wintypes
import gencb
import winmesg
import winclass
import string

# ====================
# The problem with long ints: The builtin long->int conversion fails
# for longs with the 31st bit set, because they are technically
# outside the range of a signed 32-bit integer.  However, many times
# we need to be able to pass the Windows API 32-bit quantities with
# this bit set.  So we need a way to convert from an unsigned long
# greater than 1<<31 (0x8000.0000) to its equivalent two's complement
# negative signed integer.
# I won't claim this is the best way, the only other way that comes to
# mind is to convert the number to a string and eval it (ick).
# (int(n>>1)<<1)+int(n&1)
# ====================

def safe_long (n):
	return (int(n>>1)<<1)+int(n&1)

# Window style constants are 32 bits - the first 16 flags are reserved
# for generic window styles - styles that might apply to any window,
# regardless of type.  The lower 16 are window-type-specific.

# Window style constants.
WS_OVERLAPPED	= 0x00000000L
WS_POPUP		= 0x80000000L
WS_CHILD		= 0x40000000L
WS_MINIMIZE		= 0x20000000L
WS_VISIBLE		= 0x10000000L
WS_DISABLED		= 0x08000000L
WS_CLIPSIBLINGS	= 0x04000000L
WS_CLIPCHILDREN	= 0x02000000L
WS_MAXIMIZE		= 0x01000000L
WS_CAPTION		= 0x00C00000L
WS_BORDER		= 0x00800000L
WS_DLGFRAME		= 0x00400000L
WS_VSCROLL		= 0x00200000L
WS_HSCROLL		= 0x00100000L
WS_SYSMENU		= 0x00080000L
WS_THICKFRAME	= 0x00040000L
WS_GROUP		= 0x00020000L
WS_TABSTOP		= 0x00010000L
WS_MINIMIZEBOX	= 0x00020000L
WS_MAXIMIZEBOX	= 0x00010000L
WS_TILED		= WS_OVERLAPPED
WS_ICONIC		= WS_MINIMIZE
WS_SIZEBOX		= WS_THICKFRAME

WS_POPUP = safe_long (WS_POPUP)

WS_OVERLAPPEDWINDOW = (
	WS_OVERLAPPED     |
	WS_CAPTION        |
	WS_SYSMENU        |
	WS_THICKFRAME     |
	WS_MINIMIZEBOX    |
	WS_MAXIMIZEBOX
	)

WS_POPUPWINDOW = (
	WS_POPUP          |
	WS_BORDER         |
	WS_SYSMENU
	)

WS_CHILDWINDOW = WS_CHILD


#  Extended Window Styles 
WS_EX_DLGMODALFRAME     = 0x00000001L
WS_EX_NOPARENTNOTIFY    = 0x00000004L
WS_EX_TOPMOST           = 0x00000008L
WS_EX_ACCEPTFILES       = 0x00000010L
WS_EX_TRANSPARENT       = 0x00000020L
WS_EX_MDICHILD          = 0x00000040L
WS_EX_TOOLWINDOW        = 0x00000080L
WS_EX_WINDOWEDGE        = 0x00000100L
WS_EX_CLIENTEDGE        = 0x00000200L
WS_EX_CONTEXTHELP       = 0x00000400L

WS_EX_RIGHT             = 0x00001000L
WS_EX_LEFT              = 0x00000000L
WS_EX_RTLREADING        = 0x00002000L
WS_EX_LTRREADING        = 0x00000000L
WS_EX_LEFTSCROLLBAR     = 0x00004000L
WS_EX_RIGHTSCROLLBAR    = 0x00000000L

WS_EX_CONTROLPARENT     = 0x00010000L
WS_EX_STATICEDGE        = 0x00020000L
WS_EX_APPWINDOW         = 0x00040000L

WS_EX_OVERLAPPEDWINDOW  = (WS_EX_WINDOWEDGE | WS_EX_CLIENTEDGE)
WS_EX_PALETTEWINDOW     = (WS_EX_WINDOWEDGE | WS_EX_TOOLWINDOW | WS_EX_TOPMOST)


# Does (safe_long(BIG) | SMALL) == safe_long (BIG|SMALL) ?? [it appears to]
CW_USEDEFAULT = safe_long (0x80000000L)

DEFAULT_WINDOW_STYLE = WS_OVERLAPPEDWINDOW

# ShowWindow() Commands
SW_HIDE             = 0
SW_SHOWNORMAL       = 1
SW_NORMAL           = 1
SW_SHOWMINIMIZED    = 2
SW_SHOWMAXIMIZED    = 3
SW_MAXIMIZE         = 3
SW_SHOWNOACTIVATE   = 4
SW_SHOW             = 5
SW_MINIMIZE         = 6
SW_SHOWMINNOACTIVE  = 7
SW_SHOWNA           = 8
SW_RESTORE          = 9
SW_SHOWDEFAULT      = 10
SW_MAX              = 10

#
# GetWindow() Constants
#
GW_HWNDFIRST        = 0
GW_HWNDLAST         = 1
GW_HWNDNEXT         = 2
GW_HWNDPREV         = 3
GW_OWNER            = 4
GW_CHILD            = 5
GW_MAX              = 5

cstring = windll.cstring

gdi32	 = windll.module ('gdi32')
user32	 = windll.module ('user32')

# returns LOWORD, HIWORD
def SPLIT_WORD (x):
	# fails for signed high-word
	#return (x&0xffff, (x>>16))
	return ((x<<16)>>16), (x>>16)

# Use a map to associate HWND's with window objects.  Message
# handlers are explicitly searched for by name, and called.
# How's THAT for dynamic!

try:
	window_map
	print 'window map already defined!!!'
except NameError:
	window_map = {}

# The bottom of the window class hierarchy.
# Want to be able to create one of these objects given
# nothing more than an HWND, so that we can manipulate
# windows not implemented from python.
class window:

	hwnd = 0

	def __init__ (self, hwnd=None):
		if hwnd:
			self.create (hwnd)

	def create (self, hwnd):
		global window_map
		if hwnd:
			window_map[hwnd] = self
			self.hwnd = hwnd
		else:
			raise ValueError, "create() called with NULL handle"
		# bad place for this... maybe should be a <get_paintstruct> ?
		self.paintstruct = wintypes.PAINTSTRUCT()
		return self

	def get_handle (self):
		return self.hwnd

	__int__ = get_handle

	def show_window (self, how=SW_SHOWDEFAULT):
		return user32.ShowWindow (self.hwnd, how)

	def update_window (self):
		return user32.UpdateWindow (self.hwnd)

	def move_window (self, x, y, w, h, repaint=1):
		return user32.MoveWindow (
			self,
			x, y,
			w, h,
			repaint
			)

	def is_window (self):
		return user32.IsWindow (self.hwnd)

	def is_window_visible (self):
		return user32.IsWindowVisible (self.hwnd)

	def is_iconic (self):
		return user32.IsIconic (self.hwnd)

	def destroy_window (self):
		return user32.DestroyWindow (self.hwnd)

	def close_window (self):
		return user32.CloseWindow (self.hwnd)

	_scratch_rect = wintypes.t_rect()

	def invalidate_rect (self, rect=0, erase=1):
		if rect:
			self._scratch_rect.rect = rect
			user32.InvalidateRect (
				self.hwnd,
				self._scratch_rect,
				erase
				)
		else:
			user32.InvalidateRect (self.hwnd, 0, erase)

	def get_client_rect (self):
		r = wintypes.t_rect()
		user32.GetClientRect (self, r)
		return r.rect

	def get_window_rect (self):
		r = wintypes.t_rect()
		user32.GetWindowRect (self, r)
		return r.rect

	def begin_paint (self, paintstruct=None):
		if paintstruct is None:
			paintstruct = self.paintstruct
		hdc = user32.BeginPaint (self, paintstruct)
		if hdc:
			return windc.DC (hdc)
		else:
			raise SystemError, "BeginPaint() failed"

	def end_paint (self, paintstruct=None):
		if paintstruct is None:
			paintstruct = self.paintstruct
		return user32.EndPaint (self, paintstruct)

	def get_clip_rects (self):
		region = wintypes.RGNDATA()
		region.size = region._size
		hrgn = gdi32.CreateRectRgn (0,0,1,1)
		region_type = user32.GetUpdateRgn (self.hwnd, hrgn, 0)
		retval = gdi32.GetRegionData (
			hrgn,
			region._size,
			region.address()
			)
		if retval == 0:
			print 'GetRegionData failed!'
		elif retval > region._size:
			print 'GetRegionData needed a bigger buffer (%d bytes)' % retval
		#print region
		return region.rects

	def get_dc (self):
		return windc.DC (user32.GetDC (self))

	def get_window_dc (self):
		return windc.DC (user32.GetWindowDC (self))

	def get_window (self, cmd=GW_OWNER):
		h = user32.GetWindow (self, cmd)
		if window_map.has_key (h):
			return window_map[h]
		else:
			return h

	def get_parent (self):
		h = user32.GetParent (self)
		if window_map.has_key (h):
			return window_map[h]
		else:
			return h

	def send_message (self, message, wparam, lparam):
		return user32.SendMessage (self, message, wparam, lparam)

	def set_window_long (self, offset, value):
		return user32.SetWindowLong (self, offset, value)

	def get_window_long (self, offset):
		return user32.GetWindowLong (self, offset)

	def get_style (self):
		return self.get_window_long (GWL_STYLE)

	def set_style (self, new_style):
		return self.set_window_long (GWL_STYLE, new_style)

	def set_window_text (self, text):
		#self.send_message (winmesg.messages.WM_SETTEXT, 0, cstring (text))
		return user32.SetWindowText (self, cstring (text))

	def set_parent (self, other):
		return user32.SetParent (self, other)

	def set_capture (self):
		return user32.SetCapture (self)
	
	def set_focus (self):
		return user32.SetFocus (self)

	def post_quit_message (self):
		return user32.PostQuitMessage (self)

	def release_capture (self):
		return user32.ReleaseCapture ()

	def scroll_window (self,
					   dx, dy,
					   scroll_rect=0,
					   clip_rect=0):
		# it is tempting to use ScrollWindowEx here, but
		# I think it requires specifying an explicit update
		# region, kindofa pain.
		if scroll_rect:
			r = wintypes.t_rect()
			r.rect = scroll_rect
			scroll_rect = r

		return user32.ScrollWindow (
			self,
			dx, dy,
			scroll_rect,
			clip_rect
			)

	def __repr__ (self):
		return '<%s handle:%08x at %08x>' % (
			self.__class__.__name__,
			self.hwnd,
			id(self)
			)

def get_cursor_pos ():
	p = wintypes.POINT()
	user32.GetCursorPos (p)
	return p.x, p.y

def print_compact_traceback (t, v, tb):
	tbinfo = []
	while 1:
		tbinfo.append (
			tb.tb_frame.f_code.co_filename,
			tb.tb_frame.f_code.co_name,				
			str(tb.tb_lineno)
			)
		tb = tb.tb_next
		if not tb:
			break
	tbinfo = '[' + string.join (
		map (
			lambda x: string.join (x, '|'),
			tbinfo
			),
		'] ['
		) + ']'
	print 'uncaptured python exception: (%s:%s %s)' % (str(t), str(v), tbinfo)

# A single window procedure callback is used for all windows,
# and dispatched via the global window_map.

def window_procedure (hwnd, message, wparam, lparam):
	global window_map
	try:
		if window_map.has_key (hwnd):
			retval = window_map[hwnd].window_procedure (
				hwnd, message, wparam, lparam
				)
			# A measure of convenience, and safety - makes it
			# unnecessary to put it at the end of all message
			# handlers.
			if retval is None:
				return 0
			else:
				return retval
		else:
			return user32.DefWindowProc (hwnd, message, wparam, lparam)
	except:
		import sys
		print_compact_traceback (sys.exc_type, sys.exc_value, sys.exc_traceback)
		return user32.DefWindowProc (hwnd, message, wparam, lparam)

window_procedure_callback = gencb.generated_callback ('llll', window_procedure)

default_python_window_class = None
def get_default_python_window_class ():
	global default_python_window_class
	if default_python_window_class is None:
		c = winclass.window_class ('default python window class', window_procedure)
		c.register()
		default_python_window_class = c
		return c
	else:
		return default_python_window_class

# A window with its window procedure implemented in python.
class python_window (window):
	# defaults
	hwnd		= 0
	style		= WS_OVERLAPPEDWINDOW
	parent		= 0
	menu		= 0
	instance	= 0
	param		= 0
	ext_style	= 0
	x = y = w = h = CW_USEDEFAULT
	
	def __init__ (self, window_name=0, window_class=None, **keyargs):
		self.window_name = window_name
		self.window_class = window_class
		for k, v in keyargs.items():
			setattr (self, k, v)

	def create (self):
		if self.hwnd:
			raise ValueError, "Window already has a window handle! (did you call create twice?)"

		# we should also handle winclass.window_class objects here!
		if self.window_class is None:
			self.window_class = get_default_python_window_class()
		elif type(self.window_class) == type(''):
			self.window_class = cstring (self.window_class)
		
		if type (self.window_name) == type(''):
			self.window_name = cstring (self.window_name)

		global window_map
		self.hwnd = user32.CreateWindowEx (
			self.ext_style,
			self.window_class,
			self.window_name,
			self.style,
			self.x, self.y, self.w, self.h,
			self.parent,
			self.menu,
			self.instance,
			self.param
			)
		if self.hwnd:
			window.create (self, self.hwnd)
			return self
		else:
			raise SystemError, "CreateWindowEx returned NULL"

	def window_procedure (self, hwnd, message, wparam, lparam):
		handled = 0
		# we need a way to register our own message names!
		if winmesg.db.has_key (message):
			names = winmesg.db[message]
			for name in names:
				if hasattr (self, name):
					handled = getattr (self, name) (wparam, lparam)

		if not handled:
			return user32.DefWindowProc (hwnd, message, wparam, lparam)
		else:
			return handled

	white_color = windc.get_sys_color (windc.COLOR_WINDOW)

	def WM_ERASEBKGND (self, wparam, lparam):
		dc = windc.DC (wparam)
		dc.fill_solid_rect (dc.get_clip_box(), self.white_color)
		return 1

	def WM_DESTROY (self, wparam, lparam):
		# This default destroy handler will do a PostQuitMessage if
		# this is the very last window.  This usually does the right
		# thing, but to be on the safe side you should provide at
		# least one top-level WM_DESTROY override, in case there are
		# any orphan windows (such as hidden pop-ups) that may be
		# lying around.
		global window_map
		del window_map[self.hwnd]
		self.hwnd = 0
		self.parent = None
		if not len(window_map):
			self.post_quit_message()

	def __repr__ (self):
		return '<%s: hwnd:%08x name:%s at %x>' % (
			self.__class__.__name__,
			self.hwnd,
			self.window_name,
			id(self)
			)

# From the SUBCLASS sample docs:
# 
# ==================================================
# The standard subclassing technique is to replace the window
# procedure in the window structure by using:
# 
# 	SetWindowLong (hwnd, GWL_WNDPROC, (LONG) SubclassWndProc);
#
# In the SUBCLASS sample, the old window procedure is also saved in a
# structure pointed to by the user data. Thus, any functionality can
# be added to various classes of windows without having to know the
# name of the original class.
# ==================================================
#
# The end result here is a 'subclassed' window type (say, an edit
# control, a tree control, etc...) where we can hook, capture, or
# override events.

# Window field offsets for GetWindowLong()

GWL_WNDPROC         = -4
GWL_HINSTANCE       = -6
GWL_HWNDPARENT      = -8
GWL_STYLE           = -16
GWL_EXSTYLE         = -20
GWL_USERDATA        = -21
GWL_ID              = -12

# maybe this should be a mixin?

class subclassed_window (python_window):
	def create (self):
		hwnd = python_window.create (self)
		# save away the original window procedure
		self.original_wndproc = user32.GetWindowLong (
			hwnd,
			GWL_WNDPROC
			)
		# then point to the system-wide wndproc...
		user32.SetWindowLong (
			hwnd,
			GWL_WNDPROC,
			window_procedure_callback.address
			)
		return hwnd

 	def window_procedure (self, hwnd, message, wparam, lparam):
		# we need a way to register our own message names!
		handled = 0
		if winmesg.db.has_key (message):
			names = winmesg.db[message]
			for name in names:
				if hasattr (self, name):
					handled = 1
					return getattr (self, name) (wparam, lparam)
		if not handled:
			# it is possible to call the original window procedure
			# directly, but it doesn't always work.  some window classes
			# return a 'handle' to a procedure, which CallWindowProc()
			# knows how to interpret.
			return user32.CallWindowProc (
				self.original_wndproc,
				self,
				message,
				wparam,
				lparam
				)
